# -*- coding: utf-8 -*-
""" Корректирует таблицу:
        www_station
    ##Проставляет города, регионы и time_zone тем станциям, у которых их нет
    #Привязывает станциям регионы (области) основываясь на информации из файлов
    #    ./data/karma_export/vokzali.dbf
    а также:
        www_rtstation (для определения междустанций из одной области)
    и проставляет time_zone станциям основываясь на информации из таблицы
        www_region (time_zone у которой был проставлен из geobase)
"""

import travel.avia.admin.init_project  # noqa

import logging
import time as os_time

import pytz
from django.db import connection, transaction
from django.db.models import Q, F

from travel.avia.library.python.common.models.geo import Station, Country
from travel.avia.admin.lib.logs import create_current_file_run_log
from travel.avia.admin.www.utils.data import Storage


log = logging.getLogger(__name__)


@transaction.atomic
def run():
    log.info(u"Start www_stations.py")
    start = os_time.time()
    cursor = connection.cursor()

    # update fuzzy
    Station.objects.all().update(fuzzy_only=False)
    cursor.execute("""
    SELECT s1.id FROM www_station s1 JOIN www_rtstation rts ON rts.station_id = s1.id
        GROUP BY s1.id HAVING MIN(rts.is_fuzzy) = 1
    """)
    fuzzy_station_ids = [row[0] for row in cursor.fetchall()]

    Station.objects.filter(pk__in=fuzzy_station_ids).update(fuzzy_only=True)

    # проставим станциям регионы их городов, если не совпадают
    for s in (
        Station.objects.filter(settlement__region__isnull=False)
        .exclude(region=F('settlement__region')).select_related('settlement__region', 'region')
    ):
        old_region = s.region

        s.region = s.settlement.region

        log.info(u"Сменили стации %s %s регион с %s %s на %s %s",
                 s.id, s.title,
                 old_region and old_region.id, old_region and old_region.title,
                 s.region_id, s.region.title)

        s.save()

    # проставим станциям страны их городов, если не заполнены
    for s in (
            Station.objects.filter(settlement__country__isnull=False)
            .exclude(country=F('settlement__country')).select_related('settlement__country', 'country')
    ):
        old_country = s.country

        s.country = s.settlement.country

        log.info(u"Сменили стации %s %s страну с %s %s на %s %s",
                 s.id, s.title,
                 old_country and old_country.id, old_country and old_country.title,
                 s.country_id, s.country.title)

        s.save()

    # Setting time_zone for stations by region
    for s in Station.objects.filter(region__time_zone__isnull=False,
                                      time_zone__isnull=True):
        s.time_zone = s.region.time_zone
        s.save()
    # Проставляем станциям временные зоны их городов
    for s in Station.objects.filter(time_zone__isnull=True, settlement__time_zone__isnull=False):
        log.info(u"Станции %s %s поменяли временную зону %s на зону ее города %s",
                 s.id, unicode(s), s.time_zone, s.settlement.time_zone)
        s.time_zone = s.settlement.time_zone
        s.save()
    # Если станция не имеет временной зоны и находится в стране, которая живет по временной зоне столицы
    # и не привязана к городу, проставим ей временную зону столицы
    for s in Station.objects.exclude(country__in=Country.mast_have_region_countries()).\
             filter(settlement__isnull=True,
                    time_zone__isnull=True, country__isnull=False).\
             select_related('country'):
        capital = s.country.get_capital()
        if capital:
            log.info(u"Станции %s %s поменяли временную зону %s на зону столицы %s (%s)",
                     s.id, unicode(s), s.time_zone, capital.time_zone, unicode(capital))
            s.time_zone = capital.time_zone
            s.save()

        # Выставление признака "аэроэкспрессы" аэропортам
    Station.objects.filter(rtstation__thread__express_type='aeroexpress').update(has_aeroexpress=True)

    # Раскрываем только станции, через которые ходят маршруты,
    # и останавливаются на них
    Station.objects.all().update(hidden=True)
    cursor.execute("""
    UPDATE www_station s JOIN www_rtstation rts ON rts.station_id = s.id
        JOIN www_rthread t ON t.id = rts.thread_id
        JOIN www_route r ON r.id = t.route_id
    SET s.hidden = FALSE
    WHERE (
        rts.tz_departure IS NULL OR rts.tz_arrival IS NULL
        OR (rts.tz_arrival != rts.tz_departure)
    ) AND r.hidden = FALSE AND rts.is_technical_stop = FALSE
""")

    for s in Station.objects.filter(hidden=False, properties__isnull=False):
        s.properties.all().delete()

    # Привязка с странам
    for station in Station.objects.filter(
        Q(settlement__country__isnull=False) | Q(region__country__isnull=False),
        country=None
    ):
        if station.settlement and station.settlement.country:
            station.country = station.settlement.country
            station.save()
        elif station.region and station.region.country:
            station.country = station.region.country
            station.save()

    # Обновляем ненормальные express_id и sirena_id
    cursor.execute("UPDATE www_station SET sirena_id = NULL, express_id = NULL")
    cursor.execute("""
    UPDATE www_station s JOIN www_stationcode sc ON sc.station_id = s.id
    JOIN www_codesystem cs ON cs.id = sc.system_id AND cs.code='sirena'
    SET s.sirena_id = sc.code
    """)
    cursor.execute("""
    UPDATE www_station s JOIN www_stationcode sc ON sc.station_id = s.id
    JOIN www_codesystem cs ON cs.id = sc.system_id AND cs.code='express'
    SET s.express_id = sc.code
    """)

    cursor.execute("""
UPDATE `www_station` s JOIN `www_directionmarker` dm ON dm.station_id = s.id
    JOIN `www_direction` d ON dm.direction_id = d.id
    JOIN `www_suburbanzone` sz ON d.suburban_zone_id = sz.id
SET s.suburban_zone_id = sz.id
WHERE s.suburban_zone_id IS NULL
""")

###############################################################################

    # Проставляем станциям временные зоны, пользуясь рейсами, в которые они входят.
    current_id = 0
    get_station_sql = """SELECT s.id, s.time_zone, s.country_id FROM www_station s
                    JOIN www_rtstation rts ON rts.station_id = s.id
                    WHERE country_id IS NOT NULL AND s.id>%s
                    AND s.express_id IS NOT NULL AND NOT s.hidden""" + \
                    " AND (s.time_zone NOT IN (" + ','.join(["%s"] * len(pytz.all_timezones_set)) + ") OR s.time_zone IS NULL)" + \
                    " ORDER BY s.id LIMIT 1"

    cursor.execute(get_station_sql, [current_id] + pytz.all_timezones)

    station = cursor.fetchone()

    while station:
        current_id = station[0]
        if station[1] not in pytz.all_timezones_set:
            # ищем нитку в который входит эта станция среди электричек и поездов
            cursor.execute("""
                SELECT thread_id FROM www_rtstation
                WHERE station_id = %s AND t_type_id IN (1, 6)
                LIMIT 1
            """, [station[0]])
            thread = cursor.fetchone()
            if not thread:
                cursor.execute(get_station_sql, [current_id] + pytz.all_timezones)
                station = cursor.fetchone()
                continue

            thread_id = thread[0]

            # Извлекаем путь
            cursor.execute("""
                SELECT s.id, s.time_zone, s.country_id, s.title, s.region_id
                FROM www_station s
                JOIN www_rtstation rts ON rts.station_id=s.id
                WHERE rts.thread_id=%s
                ORDER BY rts.id""", [thread_id])

            path = [Storage(id=row[0], time_zone=row[1],
                            country_id=row[2], title=row[3], region_id=row[4])
                    for row in cursor.fetchall()]

            left_tz = None
            left_region = None
            station_list = []
            for station in path:
                new_tz = None
                new_region = None
                if station.time_zone not in pytz.all_timezones_set:
                    # Если мы совсем не знаем временной зоны,
                    # добавляем станцию в список на обновление временных зон
                    station_list.append(station)

                else:
                    # Устанавливаем новую временную зону
                    new_tz = station.time_zone
                    new_region = station.region_id

                # если новая и старая временные зоны не None и совпадают регионы проставляем
                # временные зоны станциям между ними
                if left_tz and new_tz and left_region and (left_tz == new_tz) and (left_region == new_region):
                    for station in station_list:
                        cursor.execute("UPDATE www_station SET time_zone=%s WHERE id=%s", [left_tz, station.id])
                    station_list = []

                if new_tz and (new_tz != left_tz):
                    left_tz = new_tz
                    left_region = new_region
                    station_list = []

        # Берем следующую станцию
        cursor.execute(get_station_sql, [current_id] + pytz.all_timezones)
        station = cursor.fetchone()

    log.info(u"Время работы %.0f сек", os_time.time() - start)


def main():
    create_current_file_run_log()

    run()
