var path = require('path'),
    vow = require('vow'),
    fs = require('vow-fs'),
    nodeFs = require('fs'),
    vm = require('vm'),
    npmBin = __dirname + '/node_modules/.bin/';

module.exports = function(grunt) {
    var admins = ['blue', 'red', 'purple'],
        langs = ['ru', 'en'];

    // LMD builds
    var lmdBuilds = {};
    admins.forEach(function(admin) {
        langs.forEach(function(lang) {
            var output = '../desktop.bundles/index/_' + admin + '.' + lang + '.lmd.js';

            lmdBuilds['dev' + admin + lang] = {
                build: 'base+' + admin + '+' + lang,
                options: {
                    output: output
                }
            };
            lmdBuilds['prod' + admin + lang] = {
                build: 'base+' + admin + '+' + lang + '+production',
                options: {
                    output: output
                }
            };
        });
    });

    var lintTargets = {
            app: [
                'app/**/*.js',
                'Gruntfile.js'
            ],
            blocks: [
                'common.blocks/**/*.js',
                '!common.blocks/**/*.deps.js',
                '!common.blocks/**/*.bemjson.js',
                'desktop.blocks/**/*.js',
                '!desktop.blocks/**/*.deps.js',
                '!desktop.blocks/**/*.bemjson.js',
                '!**/*.i18n/*.js'
            ]
        },
        enbTargets = nodeFs.readdirSync(path.join(__dirname, 'desktop.bundles')).map(function (dir) {
            return '/desktop.bundles/' + dir;
        });

    var vendorLibs = [
        'vendors/masonry.pkgd.min.js/index.js',
        'vendors/perfect-scrollbar/min/perfect-scrollbar-0.4.8.with-mousewheel.min.js',
        'vendors/Sortable/Sortable.js'
    ];

    var vendorCss = [
        'vendors/perfect-scrollbar/min/perfect-scrollbar-0.4.8.min.css'
    ];

    grunt.initConfig({
        jshint: {
            app: lintTargets.app,
            blocks: lintTargets.blocks,
            options: {
                jshintrc: '.jshintrc',
                reporter: require('jshint-stylish')
            }
        },

        jscs: {
            app: lintTargets.app,
            blocks: lintTargets.blocks,
            options: {
                config: '.jscs.json'
            }
        },

        lmd: lmdBuilds,

        enb: {
            dev: {
                env: {
                    XJST_ASYNCIFY: 1
                },
                targets: enbTargets
            },
            prod: {
                env: {
                    XJST_ASYNCIFY: 1,
                    YENV: 'production'
                },
                targets: enbTargets
            },
            options: {
                noLog: true
            }
        },

        shell: {
            options: {
                stdout: true,
                failOnError: true,
                execOptions: {
                    maxBuffer: 5000 * 1024,
                    env: {
                        XJST_ASYNCIFY: 1,
                        YENV: 'development'
                    }
                }
            },

            enbMake: {
                command: npmBin + 'enb make'
            },

            enbClean: {
                command: npmBin + 'enb make clean'
            },

            enbMakeProd: {
                command: npmBin + 'enb make',
                options: {
                    execOptions: {
                        maxBuffer: 5000 * 1024,
                        env: {
                            XJST_ASYNCIFY: 1,
                            YENV: 'production'
                        }
                    }
                }
            },

            vendorjs: {
                command: 'cat ' + vendorLibs.join(' ') + ' > desktop.bundles/index/_vendors.js'
            },

            vendorcss: {
                command: [
                    'desktop.bundles/common/_common.css',
                    'desktop.bundles/common/_common.ie6.css',
                    'desktop.bundles/common/_common.ie7.css',
                    'desktop.bundles/common/_common.ie8.css',
                    'desktop.bundles/common/_common.ie9.css'
                ].map(function(css) {
                    return 'cat ' + css + ' ' + vendorCss.join(' ') + ' > tmp.css; mv tmp.css ' + css;
                }).join('&&')
            }
        },

        watch: {
            js: {
                files: ['app/**/*.js'],
                tasks: ['lmdDev']
            },
            templates: {
                files: [
                    'common.blocks/**/*',
                    'desktop.blocks/**/*',
                    'desktop.bundles/**/*.bemjson.js'
                ],
                tasks: ['shell:enbMake', 'lmdDev', 'shell:vendorcss']
            }
        }
    });

    grunt.loadNpmTasks('grunt-lmd');
    grunt.loadNpmTasks('grunt-shell');
    //grunt.loadNpmTasks('grunt-enb');
    grunt.loadNpmTasks('grunt-contrib-watch');
    grunt.loadNpmTasks('grunt-contrib-jshint');
    grunt.loadNpmTasks('grunt-jscs-checker');

    grunt.registerTask('i18n', function() {
        var done = this.async();

        var findKeys = function(obj) {
            var keys = [];

            if (obj instanceof Array) {
                obj.forEach(function(i) {
                    keys.push.apply(keys, findKeys(i));
                });
                return keys;
            }

            if (obj.block === 'i-bem' && obj.elem === 'i18n' && obj.keyset === 'django') {
                keys.push(obj.key);
            }

            if (obj.content) {
                keys.push.apply(keys, findKeys(obj.content));
            }

            return keys;
        };

        var getKeys = function() {
            var root = 'desktop.bundles';

            return fs.listDir(root).then(function(bundles) {
                return vow.all(bundles.map(function(bundle) {
                    if (bundle === 'common') {
                        return [];
                    }

                    var p = path.join(root, bundle, bundle + '.bemjson.js');
                    return fs.read(p, 'utf8').then(function(data) {
                        var bemjon = vm.runInThisContext(data);
                        return findKeys(bemjon);
                    });
                }));
            });
        };

        var concatUnique = function(keys) {
            return Array.prototype.concat.apply([], keys).reduce(function(a, i) {
                if (a.indexOf(i) < 0) {
                    a.push(i);
                }
                return a;
            }, []);
        };

        var mergeKeys = function(oldDict, newKeys, lang) {
            var newDict = oldDict;

            Object.keys(oldDict).forEach(function (key) {
                if (newKeys.indexOf(key) === -1) {
                    grunt.log.writeln(lang + ': найден неиспользуемый ключ: ' + key);
                }
            });

            newKeys.forEach(function(key) {
                var value = key;

                if (key in oldDict) {
                    value = oldDict[key];
                }

                if (lang !== 'ru' && value === key) {
                    grunt.log.writeln('Нет перевода ' + lang + ' для ключа: ' + key);
                }

                newDict[key] = value;
            });

            return newDict;
        };

        getKeys().then(function(keys) {
            keys = concatUnique(keys);

            grunt.log.writeln('Found ' + keys.length + ' keys');

            vow.all(langs.map(function(lang) {
                var p = path.join('common.blocks', 'gettext', 'gettext.i18n', lang + '.js'),
                    data;

                if (nodeFs.existsSync(p)) {
                    data = require('./' + p);
                } else {
                    data = { django: {} };
                }

                mergeKeys(data.django, keys, lang);
                data = 'module.exports = ' + JSON.stringify(data, null, '    ') + ';\n';

                return fs.write(p, data);
            })).then(done);
        });
    });

    grunt.registerTask('lmdDev', admins.reduce(function(targets, admin) {
        langs.forEach(function(lang) {
            targets.push('lmd:dev' + admin + lang);
        });
        return targets;
    }, []));
    grunt.registerTask('lmdProd', admins.reduce(function(targets, admin) {
        langs.forEach(function(lang) {
            targets.push('lmd:prod' + admin + lang);
        });
        return targets;
    }, []));

    grunt.registerTask('default', [
        'jshint', 'jscs',
        'shell:enbMake', 'lmdDev',
        'shell:vendorjs', 'shell:vendorcss',
        'watch'
    ]);
    grunt.registerTask('lint', ['jshint', 'jscs']);
    grunt.registerTask('dist', ['jshint', 'jscs', 'shell:enbMakeProd', 'lmdProd', 'shell:vendorjs', 'shell:vendorcss']);

};
