var app = require('app'),
    _ = require('underscore'),
    Backbone = require('backbone'),
    Marionette = require('marionette'),
    BemItemView = require('common/bemItemView'),
    BemCompositeView = require('common/bemCompositeView'),
    BemLayout = require('common/bemLayout'),
    NewSettlementForm = require('controls/goroda/newSettlementForm');

var StationItemView = BemItemView.extend({
    template: require('templates/goroda-station'),

    templateHelpers: function() {
        return {
            adminUrl: this.model.getAdminUrl()
        };
    },

    onRender: function() {
        var settlement = this.model.get('settlement');

        if (settlement) {
            this.$el.css('color', '#ccc');
        }
    }
});

var StationEmptyView = Marionette.ItemView.extend({
    template: function() {
        return 'Нет станций';
    }
});

var StationCollectionView = Marionette.CollectionView.extend({
    itemView: StationItemView,
    emptyView: StationEmptyView
});

var CityItemView = Marionette.ItemView.extend({
    template: function(data) {
        var html = BEMHTML.apply({
            block: 'button',
            mods: { size: 's' },
            attrs: { style: 'width:100%;margin-bottom:3px;' },
            content: data.title
        });

        return html;
    },

    events: {
        'click .button': 'onClick'
    },

    onClick: function() {
        this.trigger('citySelected', this.model);
    }
});

var CitiesCollectionView = BemCompositeView.extend({
    template: BEMHTML.apply([{
        block: 'list',
        content: ''
    }, {
        block: 'button',
        mix: { block: 'js-select' },
        mods: { size: 's' },
        content: gettext('Выбрать город')
    }, '&nbsp;', {
        block: 'button',
        mix: { block: 'js-create' },
        mods: { size: 's' },
        content: gettext('Создать город')
    }]),

    itemView: CityItemView,
    itemViewContainer: '.list',

    ui: {
        selectButton: ['.js-select', 'button'],
        createButton: ['.js-create', 'button']
    },

    events: {
        'click @ui.selectButton': 'onSelect',
        'click @ui.createButton': 'onCreate'
    },

    onSelect: function() {
        var button = this.ui.selectButton,
            win;

        button.val = _.bind(function(chosenId) {
            if (chosenId) {
                $.getJSON(
                    '/admin/redmap/info/settlement/' + chosenId + '/',
                    _.bind(function(info) {
                        var model = new Backbone.Model(info);
                        this.trigger('itemview:citySelected', this, model);
                    }, this)
                );
            }
        }, this);

        win = window.open(
            '/admin/www/settlement/?_to_field=id&_popup=1',
            button.params.uniqId,
            'height=500,width=800,resizable=yes,scrollbars=yes'
        );
        win.focus();
    },

    onCreate: function() {
        var form = this._form = new NewSettlementForm({model: new Backbone.Model()});

        form.on('item:closed', function () {
            delete this._form;
        }, this);

        form.on('create', function (model) {
            this.trigger('cityCreate', model);
        }, this);

        $('body').append(form.render().el);
    },

    onClose: function () {
        if (this._form) {
            this._form.close();
        }
    }
});

var ActionsView = BemItemView.extend({
    template: BEMHTML.apply([{
        block: 'button',
        mix: { block: 'js-no-city' },
        mods: { size: 's' },
        content: gettext('Станциям без города')
    }, '&nbsp;', {
        block: 'button',
        mix: { block: 'js-all' },
        mods: { size: 's' },
        content: gettext('Всем станциям')
    }]),

    events: {
        'click .js-no-city': 'onNoCityClick',
        'click .js-all': 'onAllClick'
    },

    onNoCityClick: function() {
        this.trigger('selectedMode', 'noCity');
    },

    onAllClick: function() {
        this.trigger('selectedMode', 'all');
    }
});

var GorodaView = BemLayout.extend({
    template: require('templates/goroda-control'),

    regions: {
        stationsRegion: '.stations',
        citiesRegion: '.cities'
    },

    initialize: function(options) {
        this.stations = options.stations;
        this.cities = options.settlements;

        this.listenTo(this.stations, 'reset', this.toggle, this);
        this.listenTo(this.cities, 'reset', this.toggle, this);
    },

    onRender: function() {
        var view = new StationCollectionView({
            collection: this.stations
        });
        this.stationsRegion.show(view);

        view = new ActionsView();
        this.citiesRegion.show(view);
        this.listenTo(view, 'selectedMode', this.onModeSelected, this);
        this.toggle();
    },

    onModeSelected: function(mode) {
        var view = new CitiesCollectionView({
            collection: this.cities
        });
        this.citiesRegion.show(view);

        // Event bubbling
        this.listenTo(view, 'itemview:citySelected', function(view, model) {
            this.trigger('selected', model, mode);
        }, this);

        this.listenTo(view, 'cityCreate', function(model) {
            this.trigger('create', model, mode);
        }, this);
    },

    toggle: function() {
        this.selectedMode = false;

        if (this.stations.length) {
            this.$el.show();
        } else {
            this.$el.hide();
        }
    }
});


function Goroda() {
    this.stations = new Backbone.Collection();
    this.settlements = new Backbone.Collection();
}

_.extend(Goroda.prototype, {
    _createPoly: function() {
        // Создаем многоугольник без вершин.
        var poly = new ymaps.Polygon([], {}, {
            editorDrawingCursor: 'crosshair',
            strokeWidth: 3,
            strokeColor: '#f00',
            stopDrawingByDblClick: true,
            drawOver: true
        });

        poly.events.add('geometrychange', _.debounce(_.bind(this.onGeometryChange, this), 618));
        app.map.geoObjects.add(poly);

        return poly;
    },

    _removePoly: function() {
        this.poly.events.remove('geometrychange');
        app.map.geoObjects.remove(this.poly);
        delete this.poly;
    },

    open: function() {
        this.poly = this._createPoly();
        this.poly.editor.startDrawing();

        this.stations.reset();
        this.settlements.reset();

        this.view = new GorodaView({
            stations: this.stations,
            settlements: this.settlements
        });
        this.view.on('selected', this.select, this);
        this.view.on('create', this.create, this);

        this.view.render().$el.appendTo($('body'));
    },

    close: function() {
        this._removePoly();
        this.view.close();
        delete this.view;
    },

    onGeometryChange: function() {
        var poly = this.poly;

        if (!poly || !poly.geometry.getCoordinates().length || !poly.geometry.getCoordinates()[0].length) {
            return;
        }

        var stations = [],
            settlements = [];

        app.wwwCollection.each(function(station) {
            if (!poly.geometry.contains([station.get('lon'), station.get('lat')])) {
                return;
            }

            if (station.get('settlement')) {
                settlements.push(app.wwwSettlements.get(station.get('settlement').id));
            }

            stations.push(station);
        }, this);

        settlements = _.uniq(settlements);
        this.stations.reset(stations);
        this.settlements.reset(settlements);
    },

    _listStationIds: function (mode) {
        var stations;

        if (mode === 'all') {
            stations = this.stations;
        } else if (mode === 'noCity') {
            stations = new Backbone.Collection(this.stations.filter(function(m) {
                return !m.get('settlement');
            }));
        }

        return stations.pluck('id');
    },

    create: function (model, mode) {
        var settlement = _.clone(model.attributes);

        settlement['station_ids'] = this._listStationIds(mode);

        $.ajax({
            type: 'POST',
            url: '/admin/purple/api/settlements/',
            data: JSON.stringify(settlement),
            success: _.bind(function(data) {
                if (data && data.error) {
                    alert(data.error);
                } else {
                    app.wwwCollection.fetch({ reset: true });
                    app.trigger('alert:show', {
                        msg: gettext('Станции успешно привязаны к городу') + ' ' +  model.get('title'),
                        type: 'success'
                    });

                    this.close();
                    this.open();
                }
            }, this),
            error: function () {
                alert(gettext('Ошибка связи с сервером'));
            },
            contentType: 'application/json',
            dataType: 'json'
        });
    },

    select: function(model, mode) {
        if (!confirm(gettext('Действительно привязать станции к городу') + ' ' + model.get('title') + ' ?')) {
            return;
        }

        $.post('/admin/redmap/www_stations/goroda/',
            {'settlement_id': model.get('id'), 'sids': this._listStationIds(mode)},
            function(data) {
                if (data.status === 'ok') {
                    app.trigger('alert:show', {
                        msg: gettext('Станции успешно привязаны к городу') + ' ' +  model.get('title'),
                        type: 'success'
                    });
                    app.wwwCollection.fetch({ reset: true });
                } else {
                    alert(gettext('Ошибка. не смогли назначить город станциям'));
                }
            }, 'json'
        );

        this.close();
        // Начинаем с начала
        this.open();
    }
});


module.exports = function() {
    var gorodaButton = new ymaps.control.Button({data: { content: gettext('Города') }}),
        control = new Goroda();

    gorodaButton.events.add('select', _.bind(control.open, control));
    gorodaButton.events.add('deselect', _.bind(control.close, control));

    return gorodaButton;
};
