// По станциям и дистанции между ними умеет вычеслять время
function ARMTravelTimeCalculator(options) {
    this.incSecs = options.incSecs || 10,
    this.decSecs = options.decSecs || 10,
    this.stationTime = options.stationTime || 15, // время стоянки в секундах

    this.stations = options.stations;
    this.stationKeys = $.map(Object.keys(this.stations), function(s) {
        return parseInt(s, 10);
    });
    this.distances = options.distances;
    this._oldTimes = {};

    this.init();
}

ARMTravelTimeCalculator.prototype = {
    init: function() {
        this.stationKeys.sort(function(a, b) {
            return a - b;
        });

        this.distances.getDistance = function(toId) {
            for (var i = 0; i < this.length; i++) {
                if (this[i].to === toId) {
                    return this[i].value;
                }
            }
        };
    },

    /**
     * Расчитывает время в пути в соответствии с указанной средней скоростью
     */
    calculateTravelTimes: function(averageSpeed) {

        var times = {},
            timeValue = 0;

        for (var i = 1; i < this.stationKeys.length; i++) {
            var rtstationId = this.stationKeys[i];

            timeValue += Math.round(this.distances.getDistance(this.stations[rtstationId]) / averageSpeed);

            if (i !== 1) {
                timeValue += this.stationTime;
            }

            times[rtstationId] = [timeValue, (timeValue + this.stationTime)];
        }

        times[this.stationKeys[0]] = [0, times[this.stationKeys[1]][0]];

        return times;
    },

    /**
     * Расчитывает среднюю скорость по общему времени
     */
    calculateAverageSpeed: function(totalMinutes) {
        var pathLength = 0, // длина пути в метрах
            totalTime = parseInt(totalMinutes, 10);

        totalTime *= 60; // переводим общее время следования из минут в секунды
        totalTime -= (this.stationKeys.length - 2) * this.stationTime; // вычитаем время стоянок

        $.each(this.distances, function(i, distance) {
            pathLength += distance.value;
        });

        return pathLength / totalTime;
    },

    /**
     * Устанавливает минимально корректные времена в пути для всех нижеследующих полей.
     */
    getTimesAfterStation: function(index, prevTime) {
        index = index || 1;
        prevTime = prevTime || [0, this.stationTime];

        var times = {},
            currentStationId = this.stationKeys[index];

        var arrivalVal = parseInt(prevTime[1], 10) + 1;
        var departureVal = arrivalVal + this.stationTime;

        times[currentStationId] = [arrivalVal, departureVal];

        while (index < this.stationKeys.length - 1) {
            index++;

            arrivalVal = departureVal + 1;
            departureVal = arrivalVal + this.stationTime;

            currentStationId = this.stationKeys[index];
            times[currentStationId] = [arrivalVal, departureVal];
        }

        return times;
    },

    getTimesBeforeStation: function(index, nextTime) {
        var arrival = nextTime[0],
            departure,
            id,
            times = {};

        for (var i = index; i > 0; i--) {
            id = this.stationKeys[i];

            departure = arrival - 1;
            arrival = departure - this.stationTime;
            times[id] = [arrival, departure];
        }

        return times;
    },

    /**
     * Увеличивает время прибытия и отправления на secs секунд
     */
    incTravelTime: function(value, secs) {
        return $.map(value, function(v) {
            return v += secs;
        });
    },

    /**
     * Уменьшает значение времени прибытия и отправления на secs секунд
     */
    decTravelTime: function(value, secs) {
        return $.map(value, function(v) {
            return v -= secs;
        });
    },

    getTimes: function() {
        return this._oldTimes;
    },

    setTimes: function(times) {
        this._oldTimes = times;
    },

    incStation: function(stationId, incSecs, noIncCurrent) {
        var stationIndex = this.stationKeys.indexOf(stationId),
            times = this._oldTimes;

        incSecs = incSecs || this.incSecs;

        while (stationIndex < this.stationKeys.length) {
            if (noIncCurrent) {
                noIncCurrent = false;
                stationIndex++;
                continue;
            }
            stationId = this.stationKeys[stationIndex];
            times[stationId] = this.incTravelTime(times[stationId], incSecs);

            stationIndex++;
        }

        return times;
    },

    decStation: function(stationId) {
        var stationIndex = this.stationKeys.indexOf(stationId),
            prevIndex,
            prevId,
            prevTime,
            newTime,
            times = this._oldTimes;

        while (stationIndex < this.stationKeys.length) {
            prevIndex = stationIndex - 1;
            if (prevIndex <= 0) {
                prevTime = [0, this.stationTime];
            } else {
                prevId = this.stationKeys[prevIndex];
                prevTime = times[prevId];
            }

            stationId = this.stationKeys[stationIndex];

            newTime = this.decTravelTime(times[stationId], this.decSecs);
            if (newTime[0] > prevTime[1]) {
                times[stationId] = newTime;
            }

            stationIndex++;
        }

        return times;
    },

    getTimeByIndex: function(index) {
        var id = this.stationKeys[index];

        return this._oldTimes[id] || [0, this.stationTime];
    }
};

module.exports = ARMTravelTimeCalculator;
