var _ = require('underscore'),
    Marionette = require('marionette'),
    BemCompositeView = require('common/bemCompositeView'),
    ArmStationView = require('modules/arm/stationView'),
    timeUtil = require('time');


var EmptyView = Marionette.ItemView.extend({
    tagName: 'tr',
    attributes: {
        colspan: 5
    },
    template: '<td>' + gettext('Нет станций') + '</td>'
});

var ArmView = BemCompositeView.extend({
    template: require('templates/arm-form'),

    itemView: ArmStationView,
    emptyView: EmptyView,
    itemViewContainer: '.list',

    ui: {
        totalTime: '.js-total-time',
        averageSpeed: ['.js-average-speed', 'input'],
        targetTime: ['.js-target-time', 'input'],
        speedButton: ['.js-speed-apply', 'button'],
        timeButton: ['.js-time-apply', 'button'],
        clearButton: ['.js-clear', 'button'],
        saveButton: ['.js-save', 'button']
    },

    events: {
        'click @ui.speedButton': 'applySpeed',
        'click @ui.timeButton': 'applyTime',
        'click @ui.clearButton': 'clearTime',
        'click @ui.saveButton': 'onSave'
    },

    collectionEvents: {
        'change': 'partialRender'
    },

    initialize: function(options) {
        this.startTime = options.startTime;
        this.averageSpeed = 10;
        this.targetTime = '';
    },

    itemViewOptions: function(model, index) {
        return {
            startTime: this.startTime
        };
    },

    templateHelpers: function() {
        var totalTime = this.collection.last().get('arrival');

        if (totalTime) {
            totalTime = timeUtil.humanMinutes(totalTime);
        } else {
            totalTime = '';
        }

        return {
            averageSpeed: this.averageSpeed,
            targetTime: this.targetTime,
            totalTime: totalTime
        };
    },

    partialRender: _.debounce(function() {
        var data = this.templateHelpers();
        this.ui.averageSpeed.val(data.averageSpeed);
        this.ui.targetTime.val(data.targetTime);
        this.ui.totalTime.text(data.totalTime);
    }, 200),

    _cleanErrors: function() {
        this.ui.averageSpeed.delMod('error');
        this.ui.averageSpeed.delMod(this.ui.averageSpeed.elem('message'), 'visibility');

        this.ui.targetTime.delMod('error');
        this.ui.targetTime.delMod(this.ui.targetTime.elem('message'), 'visibility');
    },

    applySpeed: function() {
        this._cleanErrors();

        this.averageSpeed = this.ui.averageSpeed.val();
        this.targetTime = this.ui.targetTime.val();

        if (!this.averageSpeed || !Number(this.averageSpeed)) {
            this.ui.averageSpeed.setMod('error', 'yes');
            this.ui.averageSpeed.setMod(this.ui.averageSpeed.elem('message'), 'visibility', 'visible');

            return;
        }

        if (!confirm('Заменить?')) {
            return;
        }

        this.trigger('applySpeed', {
            averageSpeed: this.averageSpeed /  3.6
        });
    },

    applyTime: function() {
        this._cleanErrors();

        this.averageSpeed = this.ui.averageSpeed.val();
        this.targetTime = this.ui.targetTime.val();

        var totalTime = Number(this.targetTime);

        if (!this.targetTime || !totalTime) {
            // Вы не ввели целевое время в пути || Вы ввели некорректное целевое время
            this.ui.targetTime.setMod('error', 'yes');
            this.ui.targetTime.setMod(this.ui.targetTime.elem('message'), 'visibility', 'visible');

            return;
        }

        if (!confirm('Заменить?')) {
            return;
        }

        this.trigger('applyTime', {
            totalTime: totalTime
        });
    },

    clearTime: function() {
        this.collection.each(function(model) {
            model.set({
                arrival: 0,
                departure: 0
            });
        });

        this.render();
    },

    onSave: function() {
        if (this.collection.where({valid: false}).length) {
            alert('Времена заполнены неверно');
            return;
        }

        this.collection.each(function(model) {
            model.save();
        });
        this.trigger('close');
    }
});

module.exports = ArmView;
