var Backbone = require('backbone'),
    Marionette = require('marionette'),
    _ = require('underscore'),
    ARMTravelTimeCalculator = require('modules/arm/calculator'),
    ArmFormView = require('modules/arm/formView');

var ArmController = Marionette.Controller.extend({
    initialize: function(options) {
        this.collection = options.collection;
        this.startTime = this.collection.thread.get('scheme').split(/[, |]/)[0];
        if (this.startTime) {
            var parts = this.startTime.split(':');
            this.startTime = parseInt(parts[0], 10) * 60 + parseInt(parts[1], 10);
        }
        this.armRegion = new Marionette.Region({
            el: $('<div></div>').appendTo('body')
        });
    },

    onClose: function() {
        this.isOpen = false;
        this.armRegion.close();
        this.armRegion.$el.remove();
    },

    // Api
    show: function(options) {
        this.isOpen = true;

        Backbone.ajax({
            url: _.result(this.collection, 'url') + 'distances?thread_id=' + this.collection.thread.get('id'),
            humanError: gettext('Не удалось получить дистанции'),
            success: _.bind(this._showArm, this)
        });
    },

    applySpeed: function(options) {
        var times = this.calculator.calculateTravelTimes(options.averageSpeed);

        this._applyTimes(times);
    },

    applyTime: function(options) {
        var averageSpeed = this.calculator.calculateAverageSpeed(options.totalTime),
            times = this.calculator.calculateTravelTimes(averageSpeed);

        this._applyTimes(times);
    },

    incTime: function(options) {
        var index = this.collection.indexOf(options.model),
            times = this.calculator.incStation(index);

        this._applyTimes(times);
    },

    decTime: function(options) {
        var index = this.collection.indexOf(options.model),
            times = this.calculator.decStation(index);

        this._applyTimes(times);
    },

    // Private
    _showArm: function(distances) {
        var stations = {},
            times = {};

        // Из коллекции генерируем массивы необходимые калькулятору
        this.collection.each(function(model, i) {
            stations[i] = model.get('www_station').get('id');
            // переводим время в секунды
            times[i] = [model.get('arrival') * 60, model.get('departure') * 60];
        }, this);

        // Создание и инициализация калькулятора
        this.calculator = new ARMTravelTimeCalculator({
            stations: stations,
            distances: distances,
            stationTime: 60,
            incSecs: 60,
            decSecs: 60
        });
        this.calculator.setTimes(times);

        // Создание view
        this.view = new ArmFormView({
            collection: this.collection,
            startTime: this.startTime
        });
        this._bind();

        this.armRegion.show(this.view);
    },

    _bind: function() {
        this.listenTo(this.view, 'applySpeed', this.applySpeed, this);
        this.listenTo(this.view, 'applyTime', this.applyTime, this);
        this.listenTo(this.view, 'itemview:minus', this.incTime, this);
        this.listenTo(this.view, 'itemview:plus', this.decTime, this);
        this.listenTo(this.view, 'itemview:timeChanged', this._timeChanged, this);
        this.listenTo(this.view, 'close', this.close, this);
    },

    _applyTimes: function(times) {
        _.each(times, function(time, key) {
            var model = this.collection.at(+key),
                arrival = Math.round(time[0] / 60),
                departure = Math.round(time[1] / 60);

            if (model.get('first')) {
                arrival = null;
            } if (model.get('last')) {
                departure = null;
            }

            model.set({
                arrival: arrival,
                departure: departure
            });
        }, this);
    },

    _timeChanged: function(view, options) {
        var model = options.model,
            times = this.calculator.getTimes(),
            // переводим время в секунды
            newTime = [model.get('arrival') * 60, model.get('departure') * 60],
            stationIndex = this.collection.indexOf(model),
            noTimeModels = this.collection.where({arrival: 0}).length,
            prevTime;

        // если заполнили только одно поле,
        // то, если возможно, дополняем остальные
        if (noTimeModels === this.calculator.stationKeys.length - 1) {
            if (newTime[0] - (this.calculator.stationTime) * (stationIndex - 1) < 0) {
                alert('Невозможно заполнить предыдущие поля. Введите большее значение');
                return;
            }

            // расчитываем значения для предыдущих пустых полей
            $.extend(times, this.calculator.getTimesBeforeStation(stationIndex - 1, newTime));
            // расчитываем значения для текущего и последующих полей
            prevTime = this.calculator.getTimeByIndex(stationIndex - 1);
            $.extend(times, this.calculator.getTimesAfterStation(stationIndex, prevTime));
            this._applyTimes(times);
            return;
        }

        // если введенное время прибытия меньше предыдущего времени отправления
        prevTime = this.calculator.getTimeByIndex(stationIndex - 1);
        if (newTime[0] < prevTime[1]) {
            _.extend(times, this.calculator.getTimesAfterStation(stationIndex, prevTime));
            this._applyTimes(times);
            return;
        }

        var stopTime = newTime[1] - newTime[0];

        //корректируем время отправления, если необходимо
        if (stopTime < 60) {
            newTime[1] += 60 - stopTime;
        }

        // На сколько увеличелось время
        var incrSecs = newTime[1] - times[stationIndex][1];
        times[stationIndex] = newTime;

        if (incrSecs !== 0) {
            // Увеличиваем время для всех последующий станций
            _.extend(times, this.calculator.incStation(stationIndex, incrSecs, true));
        }

        this._applyTimes(times);
    }
});

module.exports = ArmController;
