var BemItemView = require('common/bemItemView'),
    timeUtil = require('time');


var StationView = BemItemView.extend({
    tagName: 'tr',
    template: require('templates/arm-station-item'),

    ui: {
        time: ['.js-time', 'input'],
        plus: '.js-plus',
        minus: '.js-minus'
    },

    events: {
        'blur @ui.time': 'onChange',
        'click @ui.plus': 'onPlus',
        'click @ui.minus': 'onMinus',
        'change:valid': 'toggleError'
    },

    modelEvents: {
        'change': 'render'
    },

    initialize: function(options) {
        this.startTime = options.startTime;
        this.calculator = options.calculator;
    },

    templateHelpers: function() {
        var arrival = this.model.get('arrival') || 0,
            departure = this.model.get('departure') || 0,
            absoluteTime = '',
            time;

        if (arrival && departure && departure !== (arrival + 1)) {
            time = arrival + '/' + departure;
        } else {
            time = '' + arrival;
        }

        if (this.startTime) {
            absoluteTime = (this.startTime + arrival);

            var hours = Math.floor(absoluteTime / 60),
                minutes = Math.floor(absoluteTime % 60);

            if (minutes < 10) {
                minutes = '0' + minutes;
            }
            absoluteTime = ' (' + hours + ':' + minutes + ')';
        }

        absoluteTime = timeUtil.humanMinutes(arrival) + absoluteTime;

        return {
            time: time,
            absoluteTime: absoluteTime
        };
    },

    onRender: function() {
        if (this.model.get('first')) {
            this.ui.time.setMod('disabled', 'yes');
        } else {
            this.ui.time.delMod('disabled');
        }
    },

    onChange: function() {
        var val = this.ui.time.val(),
            parts = val.split('/'),
            arrival,
            departure;

        arrival = parseInt(parts[0], 10);
        if (isNaN(arrival)) {
            arrival = 0;
        }

        if (parts.length > 1) {
            departure = parseInt(parts[1], 10);
        } else {
            departure = arrival + 1;
        }

        if (this.model.get('arrival') === arrival && this.model.get('departure') === departure) {
            return;
        }

        this.model.set({
            arrival: arrival,
            departure: departure
        });

        this.trigger('timeChanged', {
            model: this.model
        });
    },

    onPlus: function() {
        this.trigger('plus', {
            model: this.model
        });
    },

    onMinus: function() {
        this.trigger('minus', {
            model: this.model
        });
    },

    toggleError: function() {
        if (this.model.get('valid')) {
            this.ui.time.delMod('error');
        } else {
            this.ui.time.setMod('error', 'yes');
        }
    }
});

module.exports = StationView;
