var app = require('app'),
    _ = require('underscore'),
    Backbone = require('Backbone'),
    GeoCollection = require('common/geoCollection'),
    StationModel = require('modules/blue_stations/model');

var StationCollection = GeoCollection.extend({
    model: StationModel,
    comparator: 'number',

    url: function() {
        return app.config.stationsUrl;
    },

    setThread: function(thread) {
        this.thread = thread;
        this.trigger('changed:thread');
    },

    parse: function(data) {
        if (data && data.length) {
            data[0].first = true;
        }

        return data;
    },

    fetch: function(options) {
        options = options || {};
        options.data = options.data || {};
        options.data['thread_id'] = this.thread.id;

        return GeoCollection.prototype.fetch.call(this, options);
    },

    reset: function(models, options) {
        var originSilent = options.silent;

        // Ставим silent = true, чтобы каждая модель не кидала событие (вьюшки начнут тормозить)
        options.silent = true;

        // Но Backbone.Relational не очищает кэш если ресет произошёл беззвучно
        // Чистим самостоятельно (RASPADMIN-98)
        this.models.forEach(function (model) {
            Backbone.Relational.store.unregister(model);
        });

        models = GeoCollection.prototype.reset.call(this, models, options);
        this.fillMissedGeo();

        // Тригерим событие reset на всей коллекции
        if (!originSilent) {
            this.trigger('reset', this, options);
        }
        return models;
    },

    fillMissedGeo: function() {
        var bounds = this.thread.getBounds(),
            middles = [],
            begin = { lat: bounds[0][1], lon: bounds[0][0] },
            end;

        function updateMiddles() {
            // Расставить middles между begin и end
            var spansCount = (middles.length + 1),
                dx = (end.lon - begin.lon) / spansCount,
                dy = (end.lat - begin.lat) / spansCount;

            _.each(middles, function(m, index) {
                m.set('lon', ((index + 1) * dx) + begin.lon);
                m.set('lat', ((index + 1) * dy) + begin.lat);
            });
        }

        this.each(function(model, index) {
            if (model.get('lat') && model.get('lon') && model.get('placed')) {
                if (middles.length) {
                    end = model.pick(['lat', 'lon']);
                    updateMiddles();
                }
                begin = model.pick(['lat', 'lon']);
                middles = [];
            } else {
                middles.push(model);
            }
        });

        if (middles.length) {
            end = { lat: bounds[1][1], lon: bounds[1][0] };
            updateMiddles();
        }
    },

    resetUnplaced: function() {
        this.each(function(model) {
            if (!model.isWwwPlaced()) {
                model.set('placed', false);
            }
        });
    }
});

module.exports = StationCollection;
