var _ = require('underscore'),
    Backbone = require('backbone'),
    Marionette = require('marionette'),
    SettlementView = require('modules/geo_settlement/view');

var SettlementController = Marionette.Controller.extend({
    initialize: function(options) {
        this.app = options.app;

        this.listenTo(this.app, 'settlement:set', this.setSettlement, this);
        this.listenTo(this.app, 'settlement:changed', this.moveMap, this);

        var initValue = options.moduleOptions['geo_settlement'].id;

        if (initValue) {
            this._getSettlement(initValue);
        }
    },

    // Api
    setSettlement: function(options) {
        this.app.currentSettlement = options.model;

        this.app.trigger('settlement:changed', {
            model: options.model,
            silent: options.silent
        });
    },

    moveMap: function(options) {
        var self = this;

        if (options.silent || !options.model) {
            return;
        }

        if (options.model.get('lat') && options.model.get('lon')) {
            this.app.trigger('maps:setCenter', [options.model.get('lon'), options.model.get('lat')]);
        } else {
            // У очень многих городов нет координат, попробуем геокодером
            var query = [options.model.get('title')];
            if (options.model.get('region')) {
                query.push(options.model.get('region').title.replace('Москва и', '')); // хак для москвы
            }
            if (options.model.get('country')) {
                query.push(options.model.get('country').title);
            } else {
                query.push('Россия');
            }

            query = query.join(', ');

            ymaps.geocode(query, { results: 1 }).then(function (res) {
                var firstGeoObject = res.geoObjects.get(0);

                if (!firstGeoObject) {
                    alert('У города отсутствует геопривязка. Не удалось определить координаты через геокодер.');
                    return;
                }

                self.app.trigger('maps:setCenter', firstGeoObject.geometry.getCoordinates());

                self.app.trigger('alert:show', {
                    msg: 'У города отсутствует геопривязка. Использован геокодер.',
                    type: 'warning',
                    source: 'settlementGeocode',
                    autoRemove: 5000
                });
            });
        }
    },

    // Private
    _getSettlement: function(id) {
        Backbone.ajax({
            url: this.app.config.polygonsUrl + id,
            success: _.bind(this._onSuccess, this)
        });
    },

    _resetSettlement: function () {
        this.setSettlement({});
    },

    _onSuccess: function(resp) {
        var model = new Backbone.Model(resp);

        this.setSettlement({
            model: model
        });
    },

    // Блоки
    selectBlock: function(region) {
        var view = new SettlementView();

        this.listenTo(view, 'changeSelected', this._getSettlement, this);
        this.listenTo(view, 'reset', this._resetSettlement, this);
        this.listenTo(this.app, 'settlement:changed', function(options) {
            if (options.model) {
                view.set(options.model.id, options.model.get('title'));
            }
        });

        region.show(view);
    }
});

module.exports = SettlementController;
