var Backbone = require('backbone'),
    _ = require('underscore'),
    GeoCollection = require('common/geoCollection'),
    StationCollection = require('modules/blue_stations/collection'),
    MetaStationModel = require('modules/red_stations/model');


var MetaStationCollection = StationCollection.extend({
    model: MetaStationModel,

    initialize: function() {
        this.on('remove', this.reNumber, this);
        // First&Lat поддерживаем в актуальном состоянии
        this.on('add', this._updateOrder, this);
        this.on('remove', this._updateOrder, this);
        this.on('reset', this._updateOrder, this);
        this.on('sync', this._updateOrder, this);
        this.on('change', this.validateTimes, this);
    },

    _updateOrder: function() {
        var prevFirst = this.findWhere({ first: true }),
            prevLast = this.findWhere({ last: true }),
            first = this.first(),
            last = this.last();

        if (!prevFirst || prevFirst.cid !== first.cid) {
            prevFirst && prevFirst.set('first', false);
            if (first) {
                first.set({
                    first: true,
                    arrival: null
                });
                !first.isNew() && first.hasChanged('arrival') && first.save();
            }
        }

        if (!prevLast || prevLast.cid !== last.cid) {
            prevLast && prevLast.set('last', false);
            if (last) {
                last.set({
                    last: true,
                    departure: null
                });
                !last.isNew() && last.hasChanged('departure') && last.save();
            }
        }
    },

    parse: function(data) {
        var i;

        for (i = 0; i < data.length; i++) {
            data[i].number = i + 1;
        }

        return data;
    },

    reset: function() {
        return GeoCollection.prototype.reset.apply(this, arguments);
    },

    create: function(json, options) {
        var _this = this;

        options = options || {};

        if (this._insertPosition) {
            json['number'] = this._insertPosition.number;
            json[this._insertPosition.position] = this._insertPosition.id;
            options.success = function(model) {
                _this.reNumber();
                _this.setInsertion(_this._insertPosition.position, model);
            };
        } else {
            json['number'] = this.length + 1;
        }

        StationCollection.prototype.create.call(this, json, options);
    },

    setInsertion: function(position, model) {
        var number = this.indexOf(model);
        if (position === 'after') {
            number += 1;
        }

        this._insertPosition = {
            position: position,
            number: number,
            model: model,
            id: model.get('id')
        };
    },

    clearInsertion: function() {
        delete this._insertPosition;
    },

    reNumber: function() {
        var i = 1;

        this.each(function(model) {
            model.set({number: i});
            i++;
        });

        this.trigger('reNumber');
    },

    prevModel: function(model) {
        var idx = this.indexOf(model);

        if (idx < 1) {
            return;
        }

        return this.at(idx - 1);
    },

    nextModel: function(model) {
        var idx = this.indexOf(model);

        if (idx >= (this.length - 1)) {
            return;
        }

        return this.at(idx + 1);
    },

    isFirst: function(model) {
        var idx = this.indexOf(model);

        if (idx === 0) {
            return true;
        }

        return false;
    },

    isLast: function(model) {
        var idx = this.indexOf(model);

        if (idx === (this.length - 1)) {
            return true;
        }

        return false;
    },

    reorder: function(ids) {
        var _this = this;

        Backbone.ajax({
            type: 'POST',
            url: _.result(this, 'url') + 'reorder',
            data: {
                'thread_id': this.thread.id,
                'order': ids || _this.pluck('id')
            },
            success: function(data) {
                _this.reset(_this.parse(data));
            },
            humanError: gettext('Не удалось поменять порядок станций')
        });
    },

    validateTimes: function(model) {
        // Если это изменилась модель и изменилось только свойство valid модели, выходим
        if (model && model.changedAttributes &&
            _.difference(_.keys(model.changedAttributes()), ['valid']).length === 0) {
            return;
        }

        this._validateTimes();
    },

    _validateTimes: _.debounce(function () {
        this.each(function (model) {
            var prev = this.prevModel(model),
                prevDeparture = (prev && prev.get('departure')) || 0,
                arrival = model.get('arrival') || 0,
                departure = model.get('departure') || (arrival + 1),
                isValid = true;

            if (model.get('last')) { // не нужно проверять departure
                if (prevDeparture > arrival) {
                    isValid = false;
                }
            } else if (prevDeparture > arrival || (departure !== null && (arrival > departure))) {
                isValid = false;
            }

            model.set('valid', isValid);
        }, this);
    }, 200)

});

module.exports = MetaStationCollection;
