var app = require('app'),
    BemItemView = require('common/bemItemView');

var StationView = BemItemView.extend({
    tagName: 'tr',
    template: require('templates/metastation-item'),
    attributes: function() {
        return {
            'data-id': this.model.id
        };
    },

    ui: {
        title: '.js-title',
        time: ['.js-time', 'input'],
        delete: ['.js-delete', 'button']
    },

    events: {
        'mouseover @ui.title': 'onHover',
        'mouseleave @ui.title': 'onUnHover',
        'click @ui.title': 'onClick',
        'click @ui.delete': 'onDeleteClick',
        'blur @ui.time': 'onTimeChanged',
        'contextmenu @ui.title': 'showPopup'
    },

    modelEvents: {
        'change:id': '_updateDataId',
        'change:valid': 'toggleError',
        'change:arrival': 'render',
        'change:departure': 'render',
        'change:number': 'render',
        'change:last': 'render',
        'change:first': 'render'
    },

    initialize: function() {
        this.listenTo(this.model, 'hover', function() {
            this.$el.addClass('hover');
        }, this);

        this.listenTo(this.model, 'unhover', function() {
            this.$el.removeClass('hover');
        }, this);
    },

    templateHelpers: function() {
        var arrival = this.model.get('arrival') || 0,
            departure = this.model.get('departure') || 0,
            time;

        if (arrival && departure && departure !== (arrival + 1)) {
            time = arrival + '/' + departure;
        } else {
            time = '' + arrival;
        }

        return {
            time: time
        };
    },

    onRender: function() {
        if (this.model.get('first')) {
            this.ui.time.setMod('disabled', 'yes');
            this.model.set('arrival', null);
            this.model.set('departure', 0);
        } else {
            this.ui.time.delMod('disabled');
        }
    },

    onHover: function() {
        this.model.trigger('hover');
    },

    onUnHover: function() {
        this.model.trigger('unhover');
    },

    onClick: function() {
        var lon = this.model.get('lon'),
            lat = this.model.get('lat');

        if (lon && lat) {
            app.trigger('maps:setCenter', [lon, lat], app.map.getZoom(), { duration: 330 });
        }
    },

    onDeleteClick: function() {
        var text = [
            gettext('Удалить станцию'),
            '«',
            this.model.get('www_station').get('title'),
            '»',
            gettext('из мета-рейса')
        ].join(' ');

        if (!confirm(text + '?')) {
            return;
        }

        this.model.destroy();
    },

    toggleError: function() {
        if (this.model.get('valid')) {
            this.ui.time.delMod('error');
        } else {
            this.ui.time.setMod('error', 'yes');
        }
    },

    onTimeChanged: function() {
        var val = this.ui.time.val(),
            parts = val.split('/'),
            arrival,
            departure;

        arrival = parseInt(parts[0], 10);
        if (isNaN(arrival)) {
            arrival = 0;
        }

        if (this.model.get('last')) {
            departure = null;
        } else if (parts.length > 1) {
            departure = parseInt(parts[1], 10);
        } else {
            departure = arrival + 1;
        }

        this.model.set({
            arrival: arrival,
            departure: departure
        });
        !this.model.isNew() && this.model.save();
    },

    _getPopup: function() {
        if (!this.popup) {
            var _this = this;

            this.popup = $(BEMHTML.apply({
                block: 'popup',
                js: {
                    directions: 'left'
                },
                content: [
                    { elem: 'tail' },
                    {
                        elem: 'content',
                        content: [{
                            tag: 'div',
                            attrs: { style: 'margin-bottom:5px;' },
                            content: [
                                'Вставить до:', '&nbsp;',
                                {
                                    block: 'link',
                                    mix: { block: 'station-create-before' },
                                    url: '#',
                                    content: 'Создать'
                                },
                                '&nbsp;',
                                {
                                    block: 'link',
                                    mix: { block: 'station-select-before' },
                                    url: '#',
                                    content: 'Выбрать'
                                }
                            ]
                        }, {
                            tag: 'div',
                            content: [
                                'Вставить после:', '&nbsp;',
                                {
                                    block: 'link',
                                    mix: { block: 'station-create-after' },
                                    url: '#',
                                    content: 'Создать'
                                },
                                '&nbsp;',
                                {
                                    block: 'link',
                                    mix: { block: 'station-select-after' },
                                    url: '#',
                                    content: 'Выбрать'
                                }
                            ]
                        }]
                    }
                ]
            })).bem('popup');

            var handler = function(type, position) {
                return function(e) {
                    e.preventDefault();
                    _this.popup.hide();
                    _this.trigger('changeInsertion', {
                        model: _this.model,
                        type: type,
                        position: position
                    });
                };
            };

            this.popup.findBlockInside('station-create-before').domElem.on('click', handler('create', 'before'));
            this.popup.findBlockInside('station-select-before').domElem.on('click', handler('select', 'before'));
            this.popup.findBlockInside('station-create-after').domElem.on('click', handler('create', 'after'));
            this.popup.findBlockInside('station-select-after').domElem.on('click', handler('select', 'after'));
        }

        return this.popup;
    },

    _updateDataId: function() {
        this.el.dataset.id = this.model.id;
    },

    showPopup: function(e) {
        e.preventDefault();

        var popup = this._getPopup();
        popup.toggle(this.ui.title);
    }
});

module.exports = StationView;
