var _ = require('underscore'),
    Marionette = require('marionette'),
    BackboneYmaps = require('Backbone.Ymaps'),
    GeoCollection = require('common/geoCollection'),
    RegionCollection = require('modules/regions/collection'),
    PlacemarkCollection = require('modules/www_stations/placemarkCollection'),
    presets = require('modules/regions/presets'),
    NoSettlementStations = require('modules/regions/stationsCollection'),
    RegionPolygon = require('modules/regions/polygon'),
    RegionPlacemark = require('modules/regions/placemark'),
    SidebarView = require('modules/regions/sidebarView');


var RegionsController = Marionette.Controller.extend({
    initialize: function(options) {
        presets.addToStorage(options.staticUrl);

        this.app = options.app;

        this.collection = new RegionCollection();
        this.noSettlementStations = new NoSettlementStations();
        this.settlement = null;

        // Т.к. работа идёт с www станциями, только по-другому
        // то проще всего сохранить внешнее api модуля
        // тогда можно будет использовать блоки написанные для www
        this._saveWwwApi();

        this._renderMapLayout();
        this._bind();
    },

    _saveWwwApi: function() {
        var self = this;

        this.app.wwwSettlements = this.collection;
        this.app.wwwCollection = {
            // Отдаём как привязанные так и не привязанные модели
            each: function(iteratee, context) {
                self.noSettlementStations.each(iteratee, context);
                self.collection.each(function(m) {
                    m.stations.each(iteratee, context);
                });

                return self.app.wwwCollection;
            },

            findClosest: function() {
                return GeoCollection.prototype.findClosest.apply(this, arguments);
            },

            fetch: function() {
                self.updateData();
            }
        };
    },

    // API
    updateData: function() {
        if (!this._updateData) {
            var self = this;
            this._updateData = _.debounce(function() {
                self.collection.fetch();
                self.noSettlementStations.fetch();
            }, 300);
        }

        this._updateData();
    },

    setSettlement: function(id) {
        if (this.settlement && this.settlement.id === id) {
            return;
        }

        var model = this.collection.get(id);

        if (!model) {
            return;
        }

        this.settlement = model;

        this.app.trigger('regions:settlement:changed', {
            model: model
        });
    },

    resetSettlement: function () {
        this.settlement = null;
        this.app.trigger('regions:settlement:changed');
    },

    // Private
    _bind: function() {
        // Связывает внешние события с событиями модуля
        this.listenTo(this.app.maps, 'boundschange', this.updateData, this);

        // Изменение глобального города должно перерендеривать сайдбар
        this.listenTo(this.app, 'settlement:changed', function() {
            var currentSettlement = this.app.currentSettlement;

            if (currentSettlement) {
                this.setSettlement(currentSettlement.id);
            } else {
                this.resetSettlement();
            }
        }, this);

        // На случай если города пришли после установки глобального
        this.listenTo(this.collection, 'sync', function() {
            if (!this.app.currentSettlement) {
                return;
            }

            this.setSettlement(this.app.currentSettlement.id);
        }, this);

        // Не забываем перезапрашивать данные (проще всей карты сразу) при изменениях
        this.listenTo(this.collection, 'redrawNeeded', this.updateData, this);
        this.listenTo(this.noSettlementStations, 'redrawNeeded', this.updateData, this);
    },

    _renderMapLayout: function() {
        // Отображение регионов
        var layer = new BackboneYmaps.CollectionView({
            geoItem: RegionPolygon,
            collection: this.collection,
            map: this.app.map
        });

        layer.render();

        // Клик на регион = выбор его из списка
        this.listenTo(layer, 'item:selected', function(options) {
            this.app.trigger('settlement:set', {
                model: options.model,
                silent: true
            });
        });

        // Отображение непривязанных точек
        layer = new PlacemarkCollection({
            geoItem: RegionPlacemark,
            collection: this.noSettlementStations,
            map: this.app.map
        });

        layer.render();
    },

    selectBlock: function(region) {
        var view;

        this.listenTo(this.app, 'regions:settlement:changed', function(options) {
            if (!options || !options.model) {
                region.close();
                return;
            }

            view = new SidebarView({
                collection: options.model.stations
            });

            region.show(view);
        }, this);
    }
});

module.exports = RegionsController;
