var _ = require('underscore'),
    app = require('app'),
    Backbone = require('backbone'),
    BemItemView = require('common/bemItemView'),
    makeOptions = require('makeOptions');

var StationForm = BemItemView.extend({
    template: require('templates/purple-station-form'),

    ui: {
        link: '.js-id a',
        title: '.js-title',
        titleRu: '.js-title-ru',
        titleUk: '.js-title-uk',
        titleTr: '.js-title-tr',
        titleEn: '.js-title-en',
        country: ['.js-country', 'related-object'],
        region: ['.js-region', 'related-object'],
        settlement: ['.js-settlement', 'related-object'],
        stationType: ['.js-station-type', 'select'],
        majority: ['.js-majority', 'select'],
        hidden: ['.js-hidden', 'radio-button'],

        cancel: ['.js-cancel', 'button'],
        unbind: ['.js-unbind', 'button'],
        save: ['.js-save', 'button']
    },

    events: {
        'click @ui.cancel': 'cancel',
        'click @ui.unbind': 'unbind',
        'click @ui.save': 'save'
    },

    attrsMapping: {
        title: 'title',
        titleRu: 'title_ru',
        titleUk: 'title_uk',
        titleTr: 'title_tr',
        titleEn: 'title_en'
    },

    relatedAttrs: ['settlement', 'region', 'country'],

    initialize: function() {
        // Не все данные для формы есть сразу
        this._loadData();
    },

    onRender: function() {
        this.ui.link.attr('href', this.model.get('geoadmin_link'));

        _.each(this.attrsMapping, function(modelAttr, uiAttr) {
            this.ui[uiAttr].val(this.model.get(modelAttr) || '');
        }, this);

        _.each(this.relatedAttrs, function(attr) {
            var val = this.model.get(attr);
            if (val) {
                this.ui[attr].val(val.id, val.title);
            }
        }, this);

        this.ui.majority.setOptions(
            makeOptions('station_majorities'),
            this.model.get('majority_id')
        );

        this.ui.stationType.setOptions(
            makeOptions('station_types'),
            this.model.get('station_type_id')
        );

        this.ui.hidden.val(this.model.get('hidden') ? 'yes' : 'no');

        if (this.model.get('settlement')) {
            this.ui.unbind.delMod('disabled');
        } else {
            this.ui.unbind.setMod('disabled', 'yes');
        }
    },

    cancel: function() {
        this.trigger('close');
    },

    unbind: function() {
        this.model.save({
            settlement: null
        }, {
            url: this._serverUrl(),
            wait: true
        });

        this.trigger('close');
    },

    save: function() {
        var newData = {};

        _.each(this.attrsMapping, function(modelAttr, uiAttr) {
            newData[modelAttr] = this.ui[uiAttr].val();
        }, this);

        _.each(this.relatedAttrs, function(attr) {
            var val = this.ui[attr].val();
            if (!val) {
                newData[attr] = null;
            } else {
                newData[attr] = { id: val };
            }
        }, this);

        newData['majority_id'] = +this.ui.majority.val();
        newData['station_type_id'] = +this.ui.stationType.val();
        newData['hidden'] = this.ui.hidden.val() === 'yes';

        this.model.save(newData, {
            url: this._serverUrl(),
            wait: true
        });

        this.trigger('close');
    },

    _serverUrl: function () {
        return app.config.stationDetailsUrl + this.model.id + '/';
    },

    _loadData: function() {
        var self = this;

        Backbone.ajax({
            url: this._serverUrl(),
            success: function(resp) {
                self.model.set(resp);
                self.onRender();
            },
            erros: function() {
                self.trigger('close');
            }
        });
    }
});

module.exports = StationForm;
