/**
 * Панель выбора сегментов геометрии маршрута для редактирования
 *
 * @param map
 * @param options
 * @constructor
 */
geomap.control.ThreadPath = function (map, options) {
    var self = this;

    self.SEGMENT_BTN_STYLES = {
        0: 'ThreadPath-btn-danger',  // Сегмент не геокодирован
        1: 'ThreadPath-btn-success', // Сегмент геокодиррован
        2: 'ThreadPath-btn-warning'  // Сегмент ранее геокодирован, но станция передвинута
    };
    self.STATION_OPTIONS = {
        preset: "islands#darkGreenIcon"
    };
    self.CURRENT_SEGMENT_POLYLINE_OPTIONS = {
        strokeColor: '#3276b1',
        strokeWidth: 4
    };

    geomap.control.ThreadPath.superclass.constructor.apply(this, [map, options, gettext('Сегменты'), 'threadPath']);
};

extend(geomap.control.ThreadPath, geomap.control.RadioButtonPanel);

mixin(geomap.control.ThreadPath.prototype, {

    defaultStyleProvider: function (map, object) {
        var self = this;
        return 'ThreadPath-left ' + (self.val() == object ? 'ThreadPath-btn-primary' : self.SEGMENT_BTN_STYLES[object.status]);
    },

    bind: function () {
        var self = this;
        self.stationLayers = self.map.controls.stationLayers;

        if (self.stationLayers) {
            var placemarkOptions = {
                optionsProvider: function (station) {
                    return {
                        draggable: true
                    };
                }
            };
            if (self.options.station && self.options.station.placemark) {
                var options = self.options.station.placemark;
                placemarkOptions.iconContentProvider = self._wrapIfNotNull(options.iconContentProvider);
                placemarkOptions.optionsProvider = self._wrapIfNotNull(options.optionsProvider);
            }
            self.stationLayers.addAdditionalDataProviders(function () {
                var result = [];
                $.each(self.objects, function(i, s) {
                    $.each([s.station_from, s.station_to], function(j, station) {
                        if (result.indexOf(station) === -1) {
                            result.push(station);
                        }
                    });
                });
                return result;
            }, placemarkOptions);
        }

        self.map._map.events.add('boundschange', function (e) {
            self._afterChangeZoom();
        });
    },

    _wrapIfNotNull: function (provider) {
        var self = this;
        if (!provider) {
            return null;
        }
        return function (station) {
            return provider.call(self, station);
        };
    },

    route: function (force) {
        var self = this,
            segment = self.val(),
            segments = segment ? [segment] : self.objects,
            changed = false;

        self.map.readyState.notReady();
        $.each(segments, function (j, s) {
            if ((force || s.status == 0) && s.station_to.t_type == 'bus') {
                self.map.readyState.notReady();
                changed = true;

                var segmentPath = s.polyline ? s.polyline.geometry.getCoordinates() : s.path;

                var path = [geomap.utils.toPoint(s.station_from)];
                if (s.path) {
                    path = path.concat(segmentPath);
                }
                path = path.concat([geomap.utils.toPoint(s.station_to)]);

                ymaps
                    .route(path, {
                               mapStateAutoApply: true
                           })
                    .then(function (route) {
                              try {
                                  s.changed = true;
                                  s.path = geomap.utils.routeToPoints(route) || path;
                              } finally {
                                  self.map.readyState.ready();
                              }
                          },
                          function () {
                              s.path = path;
                              self.map.readyState.ready();
                          }
                );
            }
        });
        self.map.readyState.onReadyOnce(function (ready) {
            if (ready && changed) {
                self.refresh(true);
            }
        });
        self.map.readyState.ready();
    },

    save: function(refresh, all) {
            var self = this,
                map = self.map,
                segment = self.val(),
                segments = !segment || all ? self.objects : [segment];

            var data = $.map(segments, function (s) {
                s.path = s.polyline.geometry.getCoordinates();
                return {
                    station_from: s.station_from.id,
                    station_to: s.station_to.id,
                    path: s.path,
                    for_two_directions: s.for_two_directions
                };
            });

            map.readyState.notReady();
            $.ajax(
                {
                    type: "POST",
                    url: "/admin/lime/save_route_path/",
                    data: JSON.stringify(data),
                    contentType: "application/json; charset=utf-8",
                    dataType: "json",
                    success: function (data) {
                        try {
                            $.each(segments, function (i, s) {
                                s.status = 1
                                s.changed = false;
                            });
                            if (refresh !== false) {
                                self.refresh(true);
                            }
                        } finally {
                            map.readyState.ready();
                        }
                    },
                    failure: function (errMsg) {
                        alert(errMsg);
                        map.readyState.ready();
                    }
                }
            );
    },

    _afterFetchData: function () {
        var self = this;
        geomap.control.RadioButtonPanel.prototype._afterFetchData.call(self);
        self.route();
    },

    _afterRefresh: function () {
        var self = this,
            currentObject = self.val();

        $.each(self.objects, function (j, segment) {
            var path = segment.path || self._defaultPath(segment);

            var polyline = new ymaps.Polyline(path, {
                hintContent: segment.title
            }, self._segmentPolylineOptions(j, segment))

            self._addGeoObject(polyline);
            segment.polyline = polyline;

            if (currentObject != segment) {
                polyline.events.add(["click"], function (event) {
                    //checkChanges();
                    self.val(segment);
                });
            } else {
                polyline.events.add(["edgedrag", "vertexadd", "vertexdrag"], function (event) {
                    segment.changed = true;
                });
            }
        });

        if (self.geoObjects.length > 0) {
            var geoObjects = currentObject ? [currentObject.polyline] : self.geoObjects,
                bounds = geoObjects[0].geometry.getBounds();

            $.each(geoObjects, function (j, object) {
                var objectBounds = object.geometry.getBounds();

                bounds[0][0] = Math.min(bounds[0][0], objectBounds[0][0]);
                bounds[0][1] = Math.min(bounds[0][1], objectBounds[0][1]);
                bounds[1][0] = Math.max(bounds[1][0], objectBounds[1][0]);
                bounds[1][1] = Math.max(bounds[1][1], objectBounds[1][1]);
            });

            self.map._map.setBounds(bounds);

            if (self.stationLayers) {
                self.stationLayers.refresh();
            }
        }

        self._afterChangeZoom();
    },

    _afterChangeZoom: function () {
        var self = this,
            editing = self.map._map.getZoom() > 10;

        $.each(self.geoObjects, function (i, polyline) {
            var val = self.val();
            if (editing && (val && val.polyline == polyline)) {
                polyline.editor.startEditing();
            } else {
                polyline.editor.stopEditing();
            }
        });
    },

    checkChanges: function () {
//        var self = this,
//            value = self.val(),
//            segments = value ? [value] : self.objects,
//            changed = false;
//
//        $.each(segments, function (i, s) {
//            changed = changed || s.changed;
//        });
//
//        if (changed) {
//            if (confirm(value ? gettext('Сохранить сегмент?') : gettext('Сохранить нитку?'))) {
//                geomap.control.buttonPanelPresets['threadPath#save'].handler(self.map, null, false);
//            } else if (!confirm(gettext('Перейти к следующему сегменту/нитке?'))) {
//                return false;
//            }
//        }
        return true;
    },

    _beforeValueChange: function () {
        var self = this;
        self.checkChanges();
    },

    _defaultPath: function (segment) {
        return [
            geomap.utils.toPoint(segment.station_from),
            geomap.utils.toPoint(segment.station_to)
        ];
    },

    _segmentPolylineOptions: function (i, segment) {
        var width = segment.status == 1 ? 2 : 4,
            color = (i % 2) == 0 ? '#9932CC' : '#7B68EE';
        return {
            strokeColor: color,
            strokeWidth: width
        }
    },

    _defaultValue: function () {
        return null;
    },

    _render: function (options, uid, style, title) {
        var self = this,
            value = self.val(),
            checked = options.for_two_directions ? ' checked ': '',
            disabled = value == null || value == options ? '' : ' disabled ',
            toolTip = self._toolTip(options, options.for_two_directions);

        return '<button id="' + uid + '" type="button" class="btn ' + style + ' btn-sm" title="' + toolTip + '">' +
            '  <input type="checkbox" id="' + uid + '-cb" ' + checked + disabled + ' onclick="event.stopPropagation();" /> ' +
            title +
            '</button>';
    },

    _toolTip: function (options) {
        var self = this,
            status = options.status,
            for_two_directions = options.for_two_directions;

        if (status == 0) {
            return for_two_directions ? gettext('Двунаправленный сегмент без геометрии')
                : gettext('Недвунаправленный сегмент без геометрии');
        }
        if (status == 1) {
            return for_two_directions ? gettext('Двунаправленный геокодированный сегмент')
                : gettext('Недвунаправленный геокодированный сегмент');
        }
        return for_two_directions ? gettext('Двунаправленный геокодированный сегмент с передвинутой станицией')
            : gettext('Недвунаправленный геокодированный сегмент с передвинутой станицией');
    },

    _afterLayout: function () {
        var self = this;
        $.each(self.objects, function (i, buttonOptions) {
           var buttonUid = self.uid + '-' + i,
               button = $('#' + buttonUid),
               twoDirection = $('#' + buttonUid + '-cb');

           twoDirection.bind('click', function() {
               buttonOptions.for_two_directions = twoDirection.is(':checked');
               buttonOptions.changed = true;

               button.attr('title', self._toolTip(buttonOptions));
           });
        });
    }
});

mixin(geomap.control.buttonPanelPresets, {

    'threadPath#selectThread': {
        title: gettext('К нитке'),
        handler: function (map, event) {
            var self = map.controls.threadPath;
            self.val(null);
        }
    },

    'threadPath#next': {
        title: gettext('-&gt;&nbsp;сегмент'),
        handler: function (map, event) {
            geomap.utils.goToNextSegment(map, geomap.utils.predicates.alwaysTrue);
        }
    },

    'threadPath#nextUnmapped': {
        title: gettext('Сохр. и дальше'),
        handler: function (map, event) {
            var self = map.controls.threadPath;
            self.save();
            geomap.utils.goToNextSegment(map, function (segment) {
                return segment.status != 1;
            });
        }
    },

    'threadPath#save': {
        title: gettext('Сохранить'),
        handler: function (map, event, refresh) {
            var self = map.controls.threadPath;

            self.save();
        }
    },

    'threadPath#saveAll': {
        title: gettext('Сохранить'),
        handler: function (map, event, refresh) {
            var self = map.controls.threadPath;

            self.save(true, true);
        }
    },

    'threadPath#reset': {
        title: gettext('Сбросить'),
        handler: function (map, event) {
            var self = map.controls.threadPath,
                value = self.val();

            if (confirm(value ? gettext('Сбросить сегмент?') : gettext('Сбросить нитку?'))) {
                var segments = value ? [value] : self.objects;

                $.each(segments, function (i, s) {
                    s.path = self._defaultPath(s);
                    s.changed = true;
                });
                self.refresh(true);
            }
        }
    },

    'threadPath#route': {
        title: gettext('Маршрутизатор'),
        handler: function (map, event) {
            var self = map.controls.threadPath,
                value = self.val();

            if (confirm(value ? gettext('Перестроить сегмент с помощью маршрутизатора?')
                            : gettext('Перестроить нитку с помощью маршрутизатора?'))) {
                self.route(true);
            }
        }
    },

    'threadPath#remove': {
        title: gettext('Удалить'),
        handler: function (map, event) {
            var self = map.controls.threadPath,
                value = self.val();

            if (confirm(value ? gettext('Удалить сегмент?') : gettext('Удалить все сегменты нитки?'))) {
                var segments = value ? [value] : self.objects,
                    stations = $.map(segments, function (s) {
                        return {
                            station_from: s.station_from.id,
                            station_to: s.station_to.id
                        };
                    });

                map.readyState.notReady();
                $.ajax(
                    {
                        type: "POST",
                        url: "/admin/lime/remove_route_path/",
                        data: JSON.stringify(stations),
                        contentType: "application/json; charset=utf-8",
                        dataType: "json",
                        success: function (data) {
                            try {
                                $.each(segments, function (i, s) {
                                    s.status = 0
                                    s.changed = false;
                                    s.path = self._defaultPath(s);
                                });
                                self.refresh(true);
                            } finally {
                                map.readyState.ready();
                            }
                        },
                        failure: function (errMsg) {
                            alert(errMsg);
                            map.readyState.ready();
                        }
                    }
                );
            }
        }
    }

});
