# -*- coding: utf-8 -*-

import logging
import subprocess
from itertools import groupby
from datetime import date, datetime, timedelta

from django import forms
from django.contrib.auth.decorators import login_required
from django.db.models import Min, Max
from django.conf import settings
from django.http import HttpResponseRedirect
from django.shortcuts import render
from travel.avia.library.python.common.utils.httpresponses import jsonp_response
from travel.avia.admin.order.models import Partner

from travel.avia.admin.lib.utils import datetime_irange


log = logging.getLogger(__name__)


@login_required
@jsonp_response
def partner_daemon_charts(request, partner_id):
    partner = Partner.objects.get(pk=partner_id)
    charts = partner.importchart_set.all()

    agg = charts.aggregate(Min('when'), Max('when'))
    min_when, max_when = agg['when__min'], agg['when__max']

    if not min_when or not max_when:
        return []

    hist_days = 28

    min_when = max(min_when, max_when - timedelta(hist_days))

    charts = charts.filter(when__gte=min_when)

    events = filter(None, (request.GET.get('events') or '').split('|'))
    if events:
        charts = charts.filter(event__in=events)

    event_key_fun = lambda chart: chart.event
    charts = sorted(charts, key=event_key_fun)
    by_event = groupby(charts, key=event_key_fun)

    minutes = 60

    def continius_grouped(events):

        def round_by_n_minutes(dt, n):
            return datetime(dt.year, dt.month, dt.day, dt.hour, n * (dt.minute / n))

        by_n_minutes = groupby(events, key=lambda (when, value): round_by_n_minutes(when, minutes))

        d = dict((k, sum((value for when, value in group)))
                 for k, group in by_n_minutes)

        return [
            (k, d.get(k))
            for k in datetime_irange(min_when, max_when, delta=timedelta(minutes=minutes))
        ]

    events_map = {
        'queryfun': 'Запрос к партнёру',
        'got_reply': 'Ответ от партнёра',
        'got_failure': 'Получена ошибка',
        'response_error': 'HTTP-ошибка в ответе',
        'query_invalid': 'Ошибочный запрос',
        'empty_response': 'Пустой ответ',
        'order_show': 'Показ на странице покупки',
    }

    return [
        {
            'label': events_map.get(event) or event,
            # like UNIX timestamp but in milliseconds
            'data': [[(int(when.strftime('%s')) + 60 * 60 * 4) * 1000, value]
                     for when, value in continius_grouped((c.when, c.value)
                                                          for c in event_charts)]
        }
        for event, event_charts in by_event
    ]


@login_required
def top_positions(request):
    if request.method == 'POST':
        form = TopForm(request.POST)

        if form.is_valid():
            sub_data = [
                'python',
                '%s/tools/top_and_positions.py' % settings.AVIA_SCRIPTS_PATH,
                '-s%s' % form.cleaned_data['start'].strftime('%Y-%m-%d'),
                '-e%s' % form.cleaned_data['end'].strftime('%Y-%m-%d'),
                '-m%s' % form.cleaned_data['email'].split('@')[0]
            ]

            try:
                subprocess.Popen(sub_data)
                status = 'ok'

            except Exception as e:
                status = e
                log.exception()

            return HttpResponseRedirect('/admin/stats/top_positions/?status=%s' % status)

    else:
        now = datetime.now().date()
        form = TopForm(initial={
            'email': request.user.email,
            'start': date(now.year, now.month, 1),
            'end': now
        })

    return render(request, 'stats/top_positions.html', {'form': form, 'status': request.GET.get('status')})


class TopForm(forms.Form):
    email = forms.EmailField(label=u'Ваш email', max_length=100)
    start = forms.DateTimeField(label=u'Дата начала', widget=forms.DateInput)
    end = forms.DateTimeField(label=u'Дата окончания', widget=forms.DateInput)
