# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import sys
import logging
from functools import update_wrapper

from django.conf import settings
from django.contrib import admin
from django.contrib.admin.options import csrf_protect_m
from django.views.debug import ExceptionReporter
from django.http import HttpResponse
from django.template import Template, Context, RequestContext
from django.utils.translation import ugettext_lazy as _
from django.shortcuts import render_to_response, redirect
from django.contrib import messages

from travel.avia.library.python.common.utils import environment
from travel.avia.admin.lib.admin_options import RaspExportModelAdmin

from travel.avia.admin.suburban_change_notices.models import Notice, NoticeEmailTemplate, Recipient
from travel.avia.admin.suburban_change_notices.sending import send_email_by_template


log = logging.getLogger(__name__)


class Notice2ExternalDirectionInlineModelAdmin(admin.TabularInline):
    model = Notice.directions.through
    raw_id_fields = ('externaldirection',)
    verbose_name = _('Внешнее направление')
    verbose_name_plural = _('Внешние направления')


class NoticeAdmin(RaspExportModelAdmin):
    inlines = [Notice2ExternalDirectionInlineModelAdmin]
    list_display = ('title', 'start_date', 'end_date')
    list_filter = ('type',)

    search_fields = ('title', 'directions__title', 'directions__code', 'text', 'mobile_text')

    fieldsets = (
        ('', {
            'fields': ('title', 'text', 'mobile_text', 'type')
        }),
        ('Даты', {
            'fields': ('start_date', 'end_date')
        }),
    )


EXCEPTION_REPORT_TEXT_TEMPLATE = """
{% if template_info %}
Template error:
In template {{ template_info.name }}, error at line {{ template_info.line }}
   {{ template_info.message }}{% for source_line in template_info.source_lines %}{% ifequal source_line.0 template_info.line %}
   {{ source_line.0 }} : {{ template_info.before }} {{ template_info.during }} {{ template_info.after }}
{% else %}
   {{ source_line.0 }} : {{ source_line.1 }}
   {% endifequal %}{% endfor %}{% endif %}{% if frames %}
Traceback:
{% for frame in frames %}File "{{ frame.filename }}" in {{ frame.function }}
{% if frame.context_line %}  {{ frame.lineno }}. {{ frame.context_line }}{% endif %}
{% endfor %}
{% if exception_type %}Exception Type: {{ exception_type }}{% if request %} at {{ request.path_info }}{% endif %}
{% if exception_value %}Exception Value: {{ exception_value }}{% endif %}{% endif %}{% endif %}
""".strip()


class NoticeEmailTemplateAdmin(RaspExportModelAdmin):
    filter_vertical = ['zones']
    filter_horizontal = ['recipients']

    def get_urls(self):
        from django.conf.urls import url

        def wrap(view):
            def wrapper(*args, **kwargs):
                return self.admin_site.admin_view(view)(*args, **kwargs)
            return update_wrapper(wrapper, view)

        urlpatterns = super(RaspExportModelAdmin, self).get_urls()

        info = self.model._meta.app_label, self.model._meta.model_name

        urlpatterns = [
            url(r'^(.+)/preview/$', wrap(self.preview), name='%s_%s_preview' % info),
            url(r'^(.+)/send_to_me/$', wrap(self.send_to_me), name='%s_%s_send_to_me' % info),
            url(r'^(.+)/send/$', wrap(self.send), name='%s_%s_send' % info),
        ] + urlpatterns

        return urlpatterns

    def preview(self, request, pk):
        net = NoticeEmailTemplate.objects.get(pk=pk)

        today = environment.today()

        try:
            old_template_debug = settings.TEMPLATE_DEBUG
            settings.TEMPLATE_DEBUG = True

            try:
                email_body = net.get_email_body(today)
            except Exception:
                context = ExceptionReporter(request, *sys.exc_info()).get_traceback_data()
                message = Template(EXCEPTION_REPORT_TEXT_TEMPLATE).render(Context(context, autoescape=False))
                return HttpResponse(message, content_type='text/plain; charset=utf-8')

        finally:
            settings.TEMPLATE_DEBUG = old_template_debug

        return HttpResponse(email_body, content_type=net.content_type)

    def send_to_me(self, request, pk):
        net = NoticeEmailTemplate.objects.get(pk=pk)

        if request.user.email:
            fails = send_email_by_template(net, [request.user.email])
            if fails:
                message_parts = ['Не удалось послать сообщение:\n']
                message_parts += ['{}: {}'.format(_email, _error) for _email, _error in fails]
                return HttpResponse('\n'.join(message_parts), content_type='text/plain; charset=utf-8')
            else:
                return HttpResponse('Сообщение успешно отправлено', content_type='text/plain; charset=utf-8')
        else:
            return HttpResponse('Email пользователя не указан', content_type='text/plain; charset=utf-8')

    @csrf_protect_m
    def send(self, request, pk):
        opts = self.model._meta

        net = NoticeEmailTemplate.objects.get(pk=pk)

        if request.method == 'POST':
            if request.session.get('noticeemailtemplate_send_redirect_to'):
                return redirect('admin:suburban_change_notices_noticeemailtemplate_change', net.id)

            log.info('** Рассылаем уведомления по шаблону %s %s "%s" из админки **',
                     net.id, net.code, net.name)

            fails = send_email_by_template(net)

            if fails:
                message_parts = ['Не удалось послать сообщение следующим адресатам:\n']
                message_parts += ['{}: {}'.format(_email, _error) for _email, _error in fails]

                request.session['noticeemailtemplate_send_redirect_to'] = True
                return HttpResponse('\n'.join(message_parts), content_type='text/plain; charset=utf-8')

            else:
                messages.info(request, 'Сообщение успешно отправлено')

                request.session['noticeemailtemplate_send_redirect_to'] = True
                return redirect('admin:suburban_change_notices_noticeemailtemplate_change', net.id)

        else:
            request.session['noticeemailtemplate_send_redirect_to'] = False

            return render_to_response(
                'admin/suburban_change_notices/noticeemailtemplate/send_mass_mail_confirmation.html',
                {
                    'net': net,
                    'app_label': opts.app_label,
                    'opts': opts,
                    'object': net,
                },
                context_instance=RequestContext(request)
            )


admin.site.register(Notice, NoticeAdmin)
admin.site.register(NoticeEmailTemplate, NoticeEmailTemplateAdmin)
admin.site.register(Recipient)
