# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import logging

from django.core.management import BaseCommand
from django.core.mail import send_mail

from travel.avia.admin.suburban_change_notices.models import NoticeEmailTemplate
from travel.avia.admin.suburban_change_notices import settings as app_settings
from travel.avia.admin.suburban_change_notices.sending import send_email_by_template
from travel.avia.library.python.common.utils import environment
from travel.avia.admin.lib.logs import print_log_to_stdout, create_run_log, get_collector


log = logging.getLogger(__name__)


class Command(BaseCommand):
    def handle(self, *args, **options):
        create_run_log('special.management_runs.' + __name__)

        if int(options.get('verbosity', 0)) > 1:
            print_log_to_stdout()

        for template_code in args:
            template_code = template_code.strip()
            if not template_code:
                continue

            collector = get_collector('', log_level=logging.INFO)
            try:
                self.send_all(template_code)
            except Exception:
                log.exception('Ошибка при рассылке уведомлений')
            else:
                log.info('Рассылка завершена успешно')

            message = collector.get_collected()

            send_mail('Отчет о рассылке по шаблонам {}'.format(template_code), message,
                      app_settings.FROM_EMAIL, [app_settings.FROM_EMAIL])

    def send_all(self, template_code):
        today = environment.today()

        log.info('Проверяем, какие уведомления нам нужно разослать для шаблонов %s.', template_code)

        templates = NoticeEmailTemplate.objects.filter(code=template_code)
        for notice_template in templates:
            recipients = notice_template.get_recipient_emails()
            has_notices = notice_template.get_today_notices(today)

            if not recipients:
                log.info('У шаблона %s %s "%s" не указаны получатели, пропускаем',
                         notice_template.id, notice_template.code, notice_template.name)
                continue

            recipients_text = ', '.join(recipients)

            if has_notices:
                log.info('Есть изменения по шаблону %s %s "%s", для подписчиков %s',
                         notice_template.id, notice_template.code, notice_template.name, recipients_text)
            else:
                log.info('Нет изменений по шаблону %s %s "%s", для подписчиков %s',
                         notice_template.id, notice_template.code, notice_template.name, recipients_text)

        log.info('Рассылаем уведомления.')

        for notice_template in templates:
            recipients = notice_template.get_recipient_emails()
            notices = notice_template.get_today_notices(today)

            recipients_text = ', '.join(recipients)

            if recipients and notices:
                log.info('=== Рассылаем изменения по шаблону %s %s "%s", для подписчиков %s',
                         notice_template.id, notice_template.code, notice_template.name, recipients_text)

                send_email_by_template(notice_template)
