# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from datetime import timedelta

from django.db import models
from django.utils.translation import get_language, activate
from django.template import Template, Context
from django.utils.translation import ugettext_lazy as _

from travel.avia.library.python.common.utils.fields import TrimmedCharField, TrimmedTextField


CHANGE_TYPES_CHOICES = (
    ('cancel', _('отмена')),
    ('assign', _('назначение')),
    ('change', _('изменение')),
    ('other',  _('другое')),
)


class Notice(models.Model):
    """
    Объявления об изменении расписания электричек
    """

    title = TrimmedCharField(_('Заголовок'), max_length=255, null=False, blank=False)
    text = TrimmedTextField(_('Текст'), blank=False)
    mobile_text = TrimmedTextField(_('Мобильный текст'), blank=True, null=True)
    type = models.CharField(_('Тип'), choices=CHANGE_TYPES_CHOICES, blank=False, default='other', null=False,
                            max_length=100)
    start_date = models.DateField(_('Дата начала'), blank=False, null=False)
    end_date = models.DateField(_('Дата окончания'), blank=False, null=False)

    directions = models.ManyToManyField('www.ExternalDirection', blank=False)

    def __unicode__(self):
        return self.title

    class Meta:
        verbose_name = _('Объявление об изменении расписания электричек')
        verbose_name_plural = _('Объявления об изменениях расписаний электричек')
        ordering = ('start_date', 'end_date')
        db_table = 'www_suburbanschedulechangenotice'


Notice.directions.through.notice.field.column = 'suburbanschedulechangenotice_id'
Notice.directions.through.notice.field.db_column = 'suburbanschedulechangenotice_id'


CONTENT_TYPE_CHOICES = (
    ('text/plain; charset=utf-8', _('Текст')),
    ('text/html; charset=utf-8', _('HTML'))
)

DEFAULT_HTML_TEMPLATE = '''
<!doctype html>
<html>
<head>
<title>{{ subject }}</title>
<style>
li {
padding: 10px 0;
}
</style>
</head>
<body>
<h1>{{ subject }}</h1>
{% if zones %}
<ul>
{% for zone in zones %}
<li>Изменения по пригородной зоне {{zone.title}}</li>
  <ul>
    {% for notice in zone.notices %}
    <li>C {{notice.start_date}} по {{notice.end_date}}:<br> {{ notice.text }}</li>
    {% endfor %}
  </ul>
{% endfor %}
</ul>
{% else %}
<p>Изменений нет</p>
{% endif %}
</body>
</html>
'''.strip()


class Recipient(models.Model):
    email = models.EmailField(blank=False)
    comment = TrimmedCharField(_('Комментарий'), max_length=200, blank=True)

    def __unicode__(self):
        if self.comment:
            return '{} - {}'.format(self.email, self.comment)
        else:
            return self.email

    class Meta:
        verbose_name = _('Получатель')
        verbose_name_plural = _('Получатели')


EMAIL_TEMPLATE_CODE_CHOICES = (
    ('draft', _('Черновик')),
    ('every_day', _('Каджный день')),
)


class NoticeEmailTemplate(models.Model):
    name = TrimmedCharField(_('Название'), max_length=255, blank=False, null=False)
    code = TrimmedCharField(_('Тип шаблона'), choices=EMAIL_TEMPLATE_CODE_CHOICES,
                            max_length=100, blank=False, null=False, default='draft')

    days_ahead = models.IntegerField(_('Дней вперед'), blank=False, default=0)
    days_ago = models.IntegerField(_('Дней назад'), blank=False, default=0)

    subject_template = TrimmedCharField(_('Шаблон заголовка'), max_length=255, blank=False, null=False)

    template = TrimmedTextField(_('Шаблон'), default=DEFAULT_HTML_TEMPLATE)

    content_type = TrimmedCharField(_('Тип шаблона'), max_length=200, choices=CONTENT_TYPE_CHOICES,
                                    null=False, blank=False, default='text/html; charset=utf-8')

    recipients = models.ManyToManyField('suburban_change_notices.Recipient', blank=True,
                                        verbose_name=_('Получатели'))

    zones = models.ManyToManyField('www.SuburbanZone', verbose_name=_('Пригородные Зоны'), blank=True)

    def get_subject(self, today):
        old_lang = get_language()
        try:
            activate('ru')
            return Template(self.subject_template).render(Context({'today': today}))

        finally:
            activate(old_lang)

    def get_recipient_emails(self):
        return [r.email for r in self.recipients.all()]

    def get_today_notices(self, today):
        notices = Notice.objects.filter(start_date__lte=today + timedelta(self.days_ahead),
                                        end_date__gte=today - timedelta(self.days_ago))
        if self.zones.count():
            notices = notices.filter(directions__suburban_zone__in=self.zones.all()).distinct()

        return notices

    def get_email_body(self, today):
        old_lang = get_language()
        try:
            activate('ru')

            subject = self.get_subject(today)

            notices = list(self.get_today_notices(today))
            notice_by_zone = {}
            for notice in notices:
                notice.zone = notice.directions.all()[0].suburban_zone
                notice_by_zone.setdefault(notice.zone, []).append(notice)

            zones = list(notice_by_zone.keys())
            zones.sort(key=lambda z: z.title)

            for zone in zones:
                zone.notices = sorted(notice_by_zone[zone], key=lambda n: (n.start_date, n.end_date))

            return Template(self.template).render(Context({
                'zones': zones,
                'subject': subject
            }))

        finally:
            activate(old_lang)

    def __unicode__(self):
        return '{} - {}'.format(self.name, self.code)

    class Meta:
        verbose_name = _('Шаблон писема')
        verbose_name_plural = _('Шаблоны писем')
