# -*- coding: utf-8 -*-

from django.db import models
from django.utils.translation import ugettext_lazy as _

from travel.avia.library.python.common.models.special_tariffs import CacheSet
from travel.avia.library.python.common.utils.fields import CodeCharField


class SkyExpressFuelSurcharge(models.Model):
    iata_from = CodeCharField(verbose_name=_(u"ИАТА код отправления"),
                              max_length=10, null=False)

    iata_to = CodeCharField(verbose_name=_(u"ИАТА код прибытия"),
                            max_length=10, null=False)

    surcharge = models.FloatField(null=False, blank=False, verbose_name=_(u"топливный сбор"))

    def __unicode__(self):
        title_from = get_point_title_by_iata(self.iata_from)
        title_to = get_point_title_by_iata(self.iata_to)
        return u"%s %s, '%s' '%s': %.2f" % (title_from, title_to, self.iata_from, self.iata_to, self.surcharge)

    class Meta:
        verbose_name = _(u"топливный cбор СкайЭкспресса")
        verbose_name_plural = _(u"топливные сборы СкайЭкспресса")
        unique_together = (('iata_from', 'iata_to'),)


def get_point_title_by_iata(iata):
    from travel.avia.library.python.common.models.geo import Station, Settlement
    try:
        point = Station.get_by_code('iata', iata)
    except Station.DoesNotExist:
        point = None
    if point is None:
        try:
            point = Settlement.objects.get(iata=iata)
        except Settlement.DoesNotExist:
            point = None

    return (point.title if point else iata)


PARNER_CHOICES = (
    ('ufs', _(u'УФС')),
    ('tickets_ua', _(u'Tickets UA')),
)


class CacheDirection(models.Model):
    set = models.ForeignKey(CacheSet, related_name="directions")

    title = models.TextField(_(u'Направление'), null=False)

    point_from_key = CodeCharField(_(u'Точка отправления'), max_length=100, null=False)
    point_to_key = CodeCharField(_(u'Точка прибытия'), max_length=100, null=False)

    alt_link = models.TextField(_(u'Альтернативные варианты'))

    data = models.TextField(_(u'Данные'))

    partner_code = models.CharField(verbose_name=_(u"партнёр"),
                                    default='ufs',
                                    choices=PARNER_CHOICES,
                                    max_length=10)

    def __unicode__(self):
        return self.title

    class Meta:
        verbose_name = _(u"направление на странице наличия свободных билетов")
        verbose_name_plural = _(u"направления на странице наличия свободных билетов")

    @property
    def dates(self):
        return self.set.dates
