# -*- coding: utf-8 -*-

import logging

from django.conf import settings
from django.utils.translation import get_language

from travel.avia.admin.templates.bemhtml import Basis

from travel.avia.admin.xgettext.i18n import gettext, xgettext

log = logging.getLogger(__name__)

ERRORS = {
    'NO_URL': gettext(u'Укажите ссылку на файл для проверки'),
    'CANT_OPEN_URL': gettext(u'Ошибка открытия ссылки'),
    'NO_FILE': gettext(u'Приложите файл для проверки'),
}

DEFAULT_ERROR = gettext(u'Произошла нейзвестная ошибка')


class Template(Basis):
    page = 'validate'

    def __json__(self):
        return [{
            'block': 'i-global',
            'params': {
                'static-host': settings.MARKUP_URL
            }
        },
            {
                'block': 'b-page',
                'title': gettext(u'Яндекс.Расписания. Валидатор общего xml-формата'),
                'head': [
                    {
                        'elem': 'css',
                        'url': self.asset_url(self.page, 'css'),
                        'ie': False,
                    },
                    {
                        'elem': 'css',
                        'url': self.asset_url(self.page, ''),
                        'ie': True
                    },
                    {'block': 'i-jquery', 'elem': 'core'},
                    {'elem': 'js', 'url': self.asset_url(self.page, '{}.pub.js'.format(get_language())), },
                    {'elem': 'js', 'url': '//yandex.st/jquery/form/3.14/jquery.form.min.js'},
                    {'elem': 'favicon', 'url': '//yandex.st/lego/_/cUDE3--xh5j8RRcJ-hym4NC4Ueg.ico'},
                ],
                'content': [
                    {
                        'block': 'l-head',
                        'mods': {'margin': 'yes'},
                        'content': [
                            {'elem': 'g'},
                            {
                                'elem': 'l',
                                'content': {
                                    'block': 'b-head-logo',
                                    'content': {
                                        'elem': 'logo',
                                        'content': {
                                            'elem': 'link',
                                            'content': {'elem': 'img'}
                                        }
                                    }
                                }
                            },
                            {'elem': 'gl'},
                            {
                                'elem': 'c',
                                'content': [
                                    {
                                        'block': 'b-head-tabs',
                                        'mods': {'type': 'search-and-content'}
                                    },
                                    {
                                        'block': 'b-head-line',
                                        'content':
                                            {
                                                'elem': 'left',
                                                'content': {
                                                    'block': 'b-link',
                                                    'attrs': {'hideFocus': True},
                                                    'url': '//rasp.yandex.ru',
                                                    'content': {
                                                        'block': 'b-plate',
                                                        'letter': list(gettext(u'РАСПИСАНИЯ'))
                                                    }
                                                }
                                            }
                                    }
                                ]
                            },
                            {'elem': 'gr'},
                            {'elem': 'r'},
                            {'elem': 'g'}
                        ]
                    },
                    {
                        'block': 'l-page',
                        'mods': {'layout': '76-16'},
                        'content': {
                            'elem': 'row',
                            'content': [
                                {'elem': 'gap'},
                                {
                                    'elem': 'left',
                                    'content': [
                                        {
                                            'block': 'b-validate-head',
                                            'content': [
                                                {
                                                    'elem': 'title',
                                                    'content': gettext(u'Валидатор общего xml-формата')
                                                },
                                                self.get_head_text()
                                            ]
                                        },
                                        {
                                            'block': 'b-validate-form',
                                            'tag': 'form',
                                            'js': True,
                                            'attrs': {
                                                'method': 'post',
                                                'action': '',
                                                'enctype': 'multipart/form-data',
                                            },
                                            'content': [
                                                {
                                                    'elem': 'title',
                                                    'tag': 'h3',
                                                    'content': gettext(u'Выберите источник XML-данных для валидации')
                                                },
                                                {
                                                    'block': 'b-tabbed-pane',
                                                    'mods': {'preset': 'white'},
                                                    'content': [
                                                        {
                                                            'elem': 'tabs',
                                                            'content': [
                                                                {
                                                                    'elem': 'tab',
                                                                    'attrs': {
                                                                        'id': 'tab-file'
                                                                    },
                                                                    'elemMods': {'state': 'current'},
                                                                    'content': gettext(u'из файла')
                                                                },
                                                                {
                                                                    'elem': 'tab',
                                                                    'attrs': {
                                                                        'id': 'tab-url'
                                                                    },
                                                                    'content': gettext(u'по ссылке')
                                                                },
                                                                {
                                                                    'elem': 'tab',
                                                                    'attrs': {
                                                                        'id': 'tab-text'
                                                                    },
                                                                    'content': gettext(u'текст XML-файла')
                                                                }
                                                            ]
                                                        },
                                                        {
                                                            'elem': 'panels',
                                                            'content': [
                                                                {
                                                                    'elem': 'panel',
                                                                    'elemMods': {'state': 'current'},
                                                                    'content':
                                                                        {
                                                                            'elem': 'wrapper',
                                                                            'content':
                                                                                {
                                                                                    'block': 'b-form-attach',
                                                                                    'content': [
                                                                                        {
                                                                                            'block': 'b-form-button',
                                                                                            'mods': {'size': 'm', 'theme': 'grey-no-transparent-m', 'valign': 'middle', 'multiple': 'no'},
                                                                                            'type': 'file',
                                                                                            'name': 'file',
                                                                                            'content': gettext(u'Выберите файл для загрузки')
                                                                                        },
                                                                                        {
                                                                                            'elem': 'holder',
                                                                                            'content': [
                                                                                                {
                                                                                                    'block': 'b-icon'
                                                                                                },
                                                                                                {
                                                                                                    'elem': 'text',
                                                                                                    'content': gettext(u'файл не выбран')
                                                                                                },
                                                                                                {
                                                                                                    'block': 'b-icon',
                                                                                                    'mix': [{'block': 'b-form-attach', 'elem': 'reset'}]
                                                                                                }
                                                                                            ]
                                                                                        }
                                                                                    ]
                                                                                }
                                                                        }
                                                                },
                                                                {
                                                                    'elem': 'panel',
                                                                    'content':
                                                                        {
                                                                            'elem': 'wrapper',
                                                                            'content':
                                                                                {
                                                                                    'tag': 'input',
                                                                                    'attrs': {
                                                                                        'name': 'url',
                                                                                        'style': 'width: 100%; height: 20px;'
                                                                                    }
                                                                                }
                                                                        }
                                                                },
                                                                {
                                                                    'elem': 'panel',
                                                                    'content':
                                                                        {
                                                                            'elem': 'wrapper',
                                                                            'content':
                                                                                {
                                                                                    'tag': 'textarea',
                                                                                    'attrs': {
                                                                                        'name': 'text',
                                                                                        'style': 'width: 100%; height: 100px; resize: none;'
                                                                                    }
                                                                                }
                                                                        }
                                                                }
                                                            ]
                                                        }
                                                    ]
                                                },
                                                {
                                                    'elem': 'button-wrapper',
                                                    'content':
                                                        {
                                                            'block': 'b-form-button',
                                                            'attrs': {'id': 'check'},
                                                            'tabindex': 1,
                                                            'mods': {'theme': 'grey-l', 'size': 'l'},
                                                            'type': 'submit',
                                                            'content': gettext(u'Проверить')
                                                        }
                                                }
                                            ]
                                        },
                                        {'block': 'b-validate'},
                                        {
                                            'block': 'wait-for-result',
                                            'attrs': {'style': 'display: none;'},
                                            'content': [
                                                {'tag': 'h3', 'content': gettext(u'Пожалуйста, подождите, идет проверка')},
                                                {
                                                    'block': 'b-spin',
                                                    'mods': {'progress': 'yes'},
                                                },
                                            ]
                                        }
                                    ]
                                },
                                {'elem': 'gap-right'},
                                {
                                    'elem': 'right',
                                    'content': [
                                        {
                                            'attrs': {'style': 'margin: 20px 0 20px -22px;'},
                                            'content': {
                                                'block': 'b-link',
                                                'mods': {'inner': 'yes'},
                                                'url': '//rasp.yandex.ru',
                                                'content': [
                                                    {
                                                        'block': 'b-icon',
                                                        'mix': [{'block': 'b-link', 'elem': 'icon'}],
                                                        'url': '//yandex.st/lego/_/GJO_ejJ0-cSzHzvjPpRmA_Jmbso.png',
                                                        'alt': '16x16 icon'
                                                    },
                                                    {
                                                        'elem': 'inner',
                                                        'content': gettext(u'Яндекс.Расписания')
                                                    }
                                                ]
                                            }
                                        },
                                        {
                                            'block': 'b-separator',
                                            'tag': 'hr'
                                        },
                                        {
                                            'attrs': {'style': 'margin: 10px 0;'},
                                            'content': {
                                                'block': 'b-link',
                                                'url': '//ya.ru',
                                                'content': gettext(u'Описание общего xml-формата')
                                            }
                                        },
                                        {
                                            'attrs': {'style': 'margin: 10px 0;'},
                                            'content': {
                                                'block': 'b-link',
                                                'url': '//rasp.yandex.ru/info/simpleformat',
                                                'content': gettext(u'Простой формат расписаний транспорта')
                                            }
                                        },
                                        {
                                            'attrs': {'style': 'margin: 10px 0;'},
                                            'content': {
                                                'block': 'b-link',
                                                'url': '//rasp.yandex.ru/info/partnersprogram',
                                                'content': gettext(u'Партнёрская программа')
                                            }
                                        },
                                        {
                                            'attrs': {'style': 'margin: 10px 0;'},
                                            'content': {
                                                'block': 'b-link',
                                                'url': '//rasp.yandex.ru/info/collaboration',
                                                'content': gettext(u'Соглашение о сотрудничестве')
                                            }
                                        }
                                    ]
                                },
                                {'elem': 'gap'},
                            ]
                        }
                    },
                    {
                        'block': 'b-foot',
                        'content': {
                            'elem': 'layout',
                            'content': [
                                {
                                    'elem': 'layout-column',
                                    'elemMods': {'type': 'left'}
                                },
                                {
                                    'elem': 'layout-column',
                                    'content': [
                                        {
                                            'elem': 'links',
                                            'content': [
                                                {
                                                    'elem': 'link',
                                                    'attr': {'type': 'agreement'},
                                                    'url': '/agreement',
                                                    'content': gettext(u'Пользовательское соглашение')
                                                }, '<br/>',
                                                {
                                                    'elem': 'link',
                                                    'url': '/partners',
                                                    'content': gettext(u'Партнёры&nbsp;сервиса')
                                                }
                                            ]
                                        },
                                        {
                                            'elem': 'links',
                                            'content': {
                                                'elem': 'link',
                                                'content': gettext(u'Мобильная версия'),
                                                'url': '//mobile.yandex.ru/?'
                                            }
                                        }
                                    ]
                                },
                                {
                                    'elem': 'layout-column',
                                    'elemMods': {'type': 'center'},
                                    'content': [
                                        {
                                            'elem': 'links',
                                            'content': [
                                                {
                                                    'elem': 'link',
                                                    'url': '/info/faq',
                                                    'content': gettext(u'Справочник&nbsp;путешественника')
                                                }, '<br/>',
                                                {
                                                    'elem': 'link',
                                                    'url': '/info/geo',
                                                    'content': gettext(u'География&nbsp;сервиса')
                                                }
                                            ]
                                        }
                                    ]
                                },
                                {
                                    'elem': 'layout-column',
                                    'content': [
                                        {
                                            'elem': 'links',
                                            'content': [
                                                {
                                                    'elem': 'link',
                                                    'url': '/info/5',
                                                    'content': gettext(u'О&nbsp;проекте')
                                                },
                                                {
                                                    'elem': 'link',
                                                    'url': '//clubs.ya.ru/rasp',
                                                    'content': gettext(u'Клуб&nbsp;сервиса')
                                                },
                                                {
                                                    'elem': 'link',
                                                    'url': '//feedback.yandex.ru?from=rasp',
                                                    'content': gettext(u'Обратная связь')
                                                },
                                                {
                                                    'elem': 'link',
                                                    'url': '//advertising.yandex.ru/media/banner/schedule.xml?from=rasp-ru',
                                                    'content': gettext(u'Реклама')
                                                }
                                            ]
                                        }
                                    ]
                                },
                                {
                                    'elem': 'layout-column',
                                    'elemMods': {'type': 'right'},
                                    'content': [
                                        {
                                            'block': 'b-copyright',
                                            'start': 2009
                                        }
                                    ]
                                }
                            ]
                        }
                    }
                ]
            }]

    def get_head_text(self):
        return {
            'elem': 'text',
            'content': [
                {
                    'tag': 'p',
                    'content': xgettext(u"""Валидатор поможет вам проверить XML-файл для
                                        <b-link url="//rasp.yandex.ru/info/partnersprogram">партнерской программы</b-link>
                                        Яндекс.Расписания на соответствие их структуры XSD-схеме.
                                        Инструмент упростит отладку экспортного файла
                                        и позволит вашим данным быстрее попасть на сервис Яндекс.Расписания""",
                                        b_link=lambda content, url: {
                                            'block': 'b-link',
                                            'url': url,
                                            'content': content
                                        })
                },
                {
                    'tag': 'p',
                    'content': gettext(
                        u'Настоятельно рекомендуем после создания экспортного файла или внесения в него изменений обязательно проводить его проверку на валидность до передачи в Яндекс.')
                }
            ]
        }


class TemplateResult(Basis):
    page = 'validate'

    def __json__(self):
        return self.get_result_block()

    def get_result_block(self):
        state = self.context.get('state')

        result = {
            'block': 'b-validate',
            'content': [
                {
                    'elem': 'subtitle',
                    'tag': 'h3',
                    'content': gettext(u'Результаты проверки')
                },
                self.get_result_function_by_state(state)()
            ]
        }

        return result

    RESULT_FUNCTION_BY_STATE = {
        'bad': '_get_bad_result_block',
        'good': '_get_good_result_block',
        'mediocre': '_get_mediocre_result_block',
    }

    def get_result_function_by_state(self, state):
        function_name = self.RESULT_FUNCTION_BY_STATE.get(state)

        if function_name and hasattr(self, function_name):
            return getattr(self, function_name)

        return self._get_error_result_block

    def _get_bad_result_block(self):
        return {
            'elem': 'result',
            'elemMods': {'valid': 'no'},
            'content': [
                {
                    'tag': 'p',
                    'content': gettext(u'Нет')
                },
                {
                    'tag': 'p',
                    'content': gettext(u'Содержание документа, найденное под корневым элементом, имеет неверный формат.')
                },
                {
                    'tag': 'p',
                    'content': self.context.get('cysix_check_error', u'')
                },
                {
                    'tag': 'pre',
                    'content': self.context.get('log_text')
                }
            ]
        }

    def _get_good_result_block(self):
        return {
            'elem': 'result',
            'elemMods': {'valid': 'yes'},
            'content': [
                {
                    'tag': 'p',
                    'content': gettext(u'Да')
                },
                {
                    'tag': 'p',
                    'content': gettext(u'Содержание документа имеет верный формат')
                },
            ]
        }

    def _get_mediocre_result_block(self):
        return {
            'elem': 'result',
            'elemMods': {'valid': 'details'},
            'content': [
                {
                    'tag': 'p',
                    'content': gettext(u'Да, но есть недочеты')
                },
                {
                    'tag': 'p',
                    'content': gettext(u'Содержание документа, найденное под корневым элементом, имеет неверный формат.')
                },
                {
                    'tag': 'pre',
                    'content': self.context.get('log_text')
                }
            ]
        }

    def _get_error_result_block(self):
        error_text = ERRORS.get(self.context.get('error_code'), DEFAULT_ERROR)
        log_text = self.context.get('log_text')

        content = []

        if error_text:
            content.append({'tag': 'p', 'content': error_text})

        if log_text:
            log_text.append({'tag': 'pre', 'content': log_text})

        return {
            'elem': 'result',
            'elemMods': {'valid': 'no'},
            'content': content
        }
