# -*- coding: utf-8 -*-
import json
import os
from copy import deepcopy
from datetime import datetime
from functools import update_wrapper

from django import forms
from django.conf import settings
from django.contrib import admin, messages
from django.contrib.admin import SimpleListFilter
from django.contrib.admin.utils import flatten_fieldsets
from django.contrib.admin.widgets import FilteredSelectMultiple
from django.contrib.contenttypes.models import ContentType

from django.db import transaction
from django.db.models import Q
from django.forms import ModelForm, modelform_factory
from django.http import HttpResponse, HttpResponseRedirect
from django.shortcuts import get_object_or_404
from django.utils.safestring import mark_safe
from django.utils.translation import ugettext_lazy as _

from travel.avia.admin.avia_data_admin.admin import AviaCompanyInline, CompanyTariff
from travel.avia.library.python.common.models.geo import (
    CityMajority, StationMajority, Country, StationType, CompanyOffice, SettlementProperty, SettlementRelatedStations,
    SettlementNearest, StationProperty, StationPassage, StationPhone, CodeSystem, StationCode, StationExpressAlias,
    SuburbanZone, ExternalDirection, ExternalDirectionMarker, PointSynonym, PseudoRegion, ReplaceException,
    GortransCityLink, StationTerminal, Station2Settlement, SettlementCode,
    Region, Direction, DirectionMarker, Settlement, Station, DirectionTranslate, District, DirectionFromTranslate
)
from travel.avia.library.python.common.models.holidays import Holiday
from travel.avia.library.python.common.models.scripts import ScriptResult, Script
from travel.avia.library.python.common.models.service import Service
from travel.avia.library.python.common.models.team import Team
from travel.avia.library.python.common.models_utils.i18n import L_field
from travel.avia.library.python.common.models.schedule import (
    VehicleProducer, ExpressTypeLite, PlatformTranslation, AviaAlliance, RThreadType,
    Supplier, DeLuxeTrain, Company, CompanySynonym, RThread, RTStation, Route, TrainSchedulePlan
)
from travel.avia.library.python.common.models.special_tariffs import SpecialOffer
from travel.avia.library.python.common.models.tariffs import SuburbanTariff, AeroexTariff, TariffType, NotTariffTrain, ThreadTariff, Setting, TariffGroup
from travel.avia.library.python.common.models.teasers import Teaser, Page
from travel.avia.library.python.common.models.transport import TransportType, TransportSubtype, TrainPseudoStationMap, TransportSubtypeColor, TransportModel
from travel.avia.library.python.common.utils import environment
from travel.avia.library.python.common.utils.date import RunMask
from travel.avia.admin.lib.admin_options import (
    RaspExportModelAdmin, OrderingOnlyOnSmallQuerysetModelAdminMixin, model_form_time_zone_mixin, TranslationsMixin
)
from travel.avia.admin.lib.adminutils import get_rasp_admin_url
from travel.avia.admin.www.admin_models.scripts import ScriptAdmin, ScriptResultAdmin
from travel.avia.admin.www.models import Redirect, DefaultPoint
from travel.avia.admin.www.models.geo import RoutePath, RegionWhiteList
from travel.avia.admin.www.models.precalc import Suggest
from travel.avia.admin.www.models.schedule import (
    TransportScheme, Route2Company, PreHoliday, RouteImportInfo, ExpressNumber, ThreadSchedule, StationTeleportMap, FaceDirection
)
from travel.avia.admin.www.models.suggests import DefaultSuggest
from travel.avia.admin.www.models.tariffs import TicketOffice


class CommentListFilter(SimpleListFilter):
    title = (_(u'Комментарий'))
    parameter_name = 'has_comment'

    def lookups(self, request, model_admin):
        return [
            (u'y', (_(u'Да'))),
            (u'n', (_(u'Нет'))),
        ]

    def queryset(self, request, queryset):

        if self.value():
            if self.value() == 'n':
                return queryset.filter(comment__exact='')
            elif self.value() == 'y':
                return queryset.exclude(comment__exact='')

        return queryset


class CityMajorityAdmin(RaspExportModelAdmin):
    fieldsets = (
        ('Profile field', {
            'fields': ('title',)
        }),
    )
    list_display = ('title',)


admin.site.register(CityMajority, CityMajorityAdmin)


class StationMajorityAdmin(TranslationsMixin, RaspExportModelAdmin):
    fieldsets = (
        ('Profile field', {
            'fields': L_field.admin_fields(StationMajority, ['title']),
            'classes': ['collapse']
        }),
        (_(u'Новые переводы'), {'fields': ('new_L_title',)}),
    )
    list_display = ('title_ru', 'code', 'id')


admin.site.register(StationMajority, StationMajorityAdmin)


class TransportSubtypeInline(admin.TabularInline):
    model = TransportSubtype
    extra = 1
    fieldsets = (
        ('Profile field', {
            'fields': [
                'code', 't_type', 'icon', 'color',
                L_field.admin_fields(TransportSubtype, [
                    'title', 'title_plural', 'title_partner_page', 'station_name', 'station_name_plural'
                ]),
                'comment'
            ]
        }),
    )


class TransportTypeAdmin(RaspExportModelAdmin):
    inlines = [TransportSubtypeInline]
    fieldsets = (
        ('Profiled field', {
            'fields': [
                'code',
            ]
        }),

        (_(u'Переводы'), {
            'fields': L_field.admin_fields(TransportType, ['title', 'title_plural', 'title_partner_page', 'station_name', 'station_name_plural'])
        }),
    )

    list_display = ('title_ru', 'code', 'station_name_plural_ru')


admin.site.register(TransportType, TransportTypeAdmin)


class VehicleProducerAdmin(RaspExportModelAdmin):
    fieldsets = (
        ('Profile field', {
            'fields': ('title', 't_type')
        }),
    )
    list_filter = ('t_type',)
    list_display = ('title', 't_type')
    search_fields = ('title',)


admin.site.register(VehicleProducer, VehicleProducerAdmin)


class TransportModelForm(forms.ModelForm):
    def clean(self):
        cleaned_data = super(TransportModelForm, self).clean()
        t_type = cleaned_data.get('t_type')
        t_subtype = cleaned_data.get('t_subtype')
        if t_type and t_subtype and t_subtype.t_type != t_type:
            raise forms.ValidationError(_(u'Подтипу транспорта {} соответствует тип транспорта {}')
                                        .format(t_subtype, t_subtype.t_type))
        return cleaned_data


def _get_fieldset(fields):
    return (('Profile field', {'fields': fields}),)


class TransportModelAdmin(RaspExportModelAdmin):
    form = TransportModelForm

    FIELDS = (
        'title', 'title_en', 't_type', 't_subtype', 'producer', 'img', 'code',
        'code_en', 'background', 'ttx', 'descr', 'template_page', 'is_cargo',
    )
    PLANE_ADDITIONAL_FIELDS = ('plane_body_type', 'is_propeller_flight')
    PLANE_FIELDS = FIELDS[:5] + PLANE_ADDITIONAL_FIELDS + FIELDS[5:]
    fieldsets = _get_fieldset(FIELDS)
    list_display = ('title', 'title_en', 'code', 'code_en',
                    'producer', 't_type', 'template_page')
    list_filter = ('t_type', 't_subtype', 'producer')
    search_fields = ('title', 'title_en', 'code', 'ttx', 'descr')

    def get_form(self, request, obj=None, **kwargs):
        if obj and obj.t_type_id == TransportType.PLANE_ID:
            self.fieldsets = _get_fieldset(self.PLANE_FIELDS)
        else:
            self.fieldsets = _get_fieldset(self.FIELDS)

        return super(TransportModelAdmin, self).get_form(request, obj, **kwargs)


admin.site.register(TransportModel, TransportModelAdmin)


class ShowUrlOnDublicateGeoIdForm(forms.ModelForm):
    def clean__geo_id(self):
        geo_id = self.cleaned_data['_geo_id']

        if geo_id is None:
            return geo_id

        objs = self._meta.model.objects.filter(_geo_id=geo_id).exclude(pk=self.instance.pk)

        if objs:
            title = objs[0].L_title()
            url = get_rasp_admin_url(objs[0], relative_only=False)

            url = u'<a href="{url}" target="_blank">{title}</a>'.format(url=url, title=title)

            model_name = self._meta.model._meta.verbose_name

            error = _(u'{model_name} с таким Geo ID уже есть в базе: {obj_url}.')\
                .format(obj_url=url, model_name=model_name.capitalize())

            raise forms.ValidationError(mark_safe(error))

        return geo_id


class CountryForm(ShowUrlOnDublicateGeoIdForm):
    class Meta:
        model = Country
        exclude = []


class CountryAdmin(TranslationsMixin, RaspExportModelAdmin):
    form = CountryForm

    def has_delete_permission(self, request, obj=None):
        return False

    def has_add_permission(self, request):
        return False

    list_display = ('id', '_geo_id', 'title', 'title_en', 'code', 'currency', 'language')
    search_fields = ('title', 'title_en')
    readonly_fields = ('modified_at',)

    fieldsets = (
        (_(u'Последнее обновление'), {'fields': ('modified_at',)}),
        (_(u'Название'), {'fields': ['title']}),
        (_(u'Переводы'), {
            'fields': list(L_field.admin_fields(Country, ['title'])),
            'classes': ['collapse']
        }),
        (_(u'Новые переводы'), {'fields': ('new_L_title',)}),
        (_(u'Коды'), {'fields': ['_geo_id', 'code', 'code3', 'domain_zone']}),
        (_(u'Другое'), {'fields': ['language', 'currency']}),
    )


admin.site.register(Country, CountryAdmin)


class RegionForm(ShowUrlOnDublicateGeoIdForm,
                 model_form_time_zone_mixin('time_zone', additonal_choices=[(u'', u'!!!Не указана :(')])):
    class Meta:
        model = Region
        exclude = []


class RegionAdmin(TranslationsMixin, RaspExportModelAdmin):
    form = RegionForm

    def has_delete_permission(self, request, obj=None):
        return False

    def has_add_permission(self, request):
        return False

    raw_id_fields = ('country',)
    list_display = ('id', '_geo_id', 'title', 'title_en', 'time_zone', 'country', 'hidden')
    search_fields = ('title', 'title_en')
    readonly_fields = ('modified_at', '_kladr_id')

    fieldsets = (
        (_(u'Последнее обновление'), {'fields': ('modified_at',)}),
        (_(u'Название'), {'fields': ('title',)}),
        (_(u'Переводы'), {
            'fields': list(L_field.admin_fields(Region, ['title'])),
            'classes': ['collapse']
        }),
        (_(u'Новые переводы'), {'fields': ('new_L_title',)}),
        (_(u'Основные'), {
            'fields': ['country', 'time_zone', 'hidden', 'disputed_territory']
        }),
        (_(u'Коды'), {
            'fields': ['_geo_id', '_kladr_id', 'koatuu', 'agent_geo_id']
        }),
    )


admin.site.register(Region, RegionAdmin)


class SettlementPropertyAdmin(RaspExportModelAdmin):
    fieldsets = (
        ('Profile field', {
            'fields': ('title', 'description')
        }),
    )
    list_display = ('title', 'description')


admin.site.register(SettlementProperty, SettlementPropertyAdmin)


class SettlementRelatedStationsAdmin(RaspExportModelAdmin):
    raw_id_fields = ('station', 'settlement')

    fieldsets = (
        ('Profile field', {
            'fields': ('station', 'settlement')
        }),
    )
    list_display = ('station', 'settlement')
    search_fields = ('station__title', 'settlement__title')


admin.site.register(SettlementRelatedStations, SettlementRelatedStationsAdmin)


class SettlementNearestAdmin(RaspExportModelAdmin):
    raw_id_fields = ('settlement', 'nearest')
    list_display = ('settlement', 'nearest')


admin.site.register(SettlementNearest, SettlementNearestAdmin)


class SettlementNearestInline(admin.TabularInline):
    model = SettlementNearest
    fk_name = 'settlement'
    raw_id_fields = ('nearest',)
    extra = 1


class SettlementForm(ShowUrlOnDublicateGeoIdForm,
                     model_form_time_zone_mixin('time_zone', additonal_choices=[(u'', u'!!!Не указана :(')])):
    class Meta:
        model = Settlement
        exclude = []


class SettlementCodeInline(admin.TabularInline):
    model = SettlementCode


class SettlementAdmin(TranslationsMixin, RaspExportModelAdmin):
    form = SettlementForm

    def has_delete_permission(self, request, obj=None):
        return False

    def has_add_permission(self, request):
        return False

    raw_id_fields = ('country', 'region', 'district')

    fieldsets = (
        (_(u'Последнее обновление'), {'fields': ('modified_at',)}),
        (_(u'Локальные названия'), {
            'fields': [
                'title',
                'abbr_title',
            ]
        }),
        (_(u'Переводы'), {
            'fields': L_field.admin_fields(Settlement, ['title', 'abbr_title']),
            'classes': ['collapse']
        }),
        (_(u'Новые переводы'), {'fields': ['new_L_title', 'new_L_abbr_title']}),
        ('Profile field', {
            'fields': (
                'old_avia_id',
                'country', 'region', 'district', 'majority', 'suggest_order',
                'time_zone',
                'phone_info', 'phone_info_short',
                'big_city', 'hidden', 'has_urban_transport', 'has_tablo', '_geo_id', 'properties',
                'sirena_id', 'iata', 'agent_geo_id', 'koatuu', 'suburban_zone', 'type_choices',
            )
        }),
        ('Additional', {'fields': ('human_url', '_disputed_territory',), 'classes': ('collapse',)}),
        (_(u"Координаты"), {
            'classes': ('ymaps-point',),
            'fields': ('longitude', 'latitude'),
        }),
    )
    readonly_fields = ('type_choices', 'modified_at', 'old_avia_id')
    list_display = ('id', '_geo_id', 'title', 'title_en', 'majority', 'suggest_order', 'country',
                    'region', 'time_zone', 'sirena_id', 'big_city',)
    list_display_links = ('title',)
    list_filter = ('majority', 'big_city')
    search_fields = ('title', 'title_en', 'sirena_id', 'iata')
    filter_horizontal = ('properties',)
    inlines = (SettlementNearestInline, SettlementCodeInline)


admin.site.register(Settlement, SettlementAdmin)


class StationPropertyAdmin(RaspExportModelAdmin):
    fieldsets = (
        ('Profile field', {
            'fields': ('title', 'description',)
        }),
    )
    list_display = ('title',)


admin.site.register(StationProperty, StationPropertyAdmin)


class StationTypeAdmin(TranslationsMixin, RaspExportModelAdmin):
    list_display = ('name_ru', 'prefix_ru')
    fieldsets = (
        (None, {'fields': ['title_regex', 'default_for_t_type']}),
        (_(u'Переводы'), {'fields': L_field.admin_fields(StationType, ['name', 'prefix', 'railway_prefix']), 'classes': ('collapse',)}),
        (_(u'Новые переводы'), {'fields': ['new_L_name', 'new_L_prefix', 'new_L_railway_prefix']}),
    )


admin.site.register(StationType, StationTypeAdmin)


class StationCodeInline(admin.TabularInline):
    model = StationCode


class StationTerminalInline(admin.TabularInline):
    model = StationTerminal


class StationPhoneInline(admin.TabularInline):
    model = StationPhone


StationTimezoneFormMixIn = model_form_time_zone_mixin('time_zone', additonal_choices=[(u'', u'!!!Не указана :(')],
                                                      required=False)


class StationAdmin(TranslationsMixin, RaspExportModelAdmin):
    form = modelform_factory(Station, StationTimezoneFormMixIn, exclude=[])

    def has_delete_permission(self, request, obj=None):
        return False

    def has_add_permission(self, request):
        return False

    raw_id_fields = ('settlement', 'region', 'country', 'district')
    inlines = [StationTerminalInline, StationCodeInline, StationPhoneInline]

    fieldsets = (
        (_(u'Последнее обновление'), {'fields': ('modified_at',)}),
        (_(u'Локальные названия'), {
            'fields': ['title', 'popular_title', 'short_title']
        }),
        (_(u'Переводы'), {
            'fields': L_field.admin_fields(Station, ['title', 'popular_title', 'short_title'], group_by='lang'),
            'classes': ('collapse',)
        }),
        (_(u'Новые_переводы'), {
            'fields': ['new_L_title', 'new_L_popular_title', 'new_L_short_title']
        }),
        (_(u"Расположение"), {
            'fields': ('country', 'region', 'settlement', 'district', 'time_zone', 'time_zone_not_check')
        }),
        (_(u"Адрес"), {
            'fields': L_field.admin_fields(Station, ['address'], show_local_field=True),
            'classes': ('collapse',)
        }),
        (_(u"Адрес"), {
            'fields': ['new_L_address']
        }),
        (_(u"Тип"), {
            'fields': ('t_type', 'station_type')
        }),
        (_(u"Базовый автовокзал"), {
            'fields': ('is_base',)
        }),
        (_(u"Нечеткие признаки"), {
            'fields': (
                'is_fuzzy', 'is_searchable_from', 'is_searchable_to',
                'in_station_schedule', 'in_thread'
            )
        }),
        (_(u"Как добраться до города"), {
            'fields': L_field.admin_fields(Station, ['how_to_get_to_city']),
            'classes': ('collapse',)
        }),
        (_(u"Как добраться до города"), {
            'fields': ['new_L_how_to_get_to_city']
        }),
        (_(u'Не сортированные настройки'), {
            'fields': (
                'hidden', 'use_direction',
                'majority',
                'show_settlement', 'site_url',
                'show_mode',
                'not_generalize', 'suburban_zone', 'has_aeroexpress', 'near_metro',
                'map_zoom', 'tablo_state', 'tablo_state_prev', 'virtual_end', 'type_choices', 'incomplete_bus_schedule',
                'show_tablo_stat'
            )
        }),
        (_(u'Панорама и схема'), {
            'fields': ('photo', 'panorama_url', 'schema_image'),
        }),
        (_(u"Координаты"), {
            'classes': ('ymaps-point',),
            'fields': ('longitude', 'latitude'),
        }),
        (u'Meta', {
            'fields': ('meta_title', 'meta_description')
        })
    )
    list_display = ('id', 'title', 'station_type', 'majority', 'full_path_with_links', 't_type',
                    'express_id', 'sirena_id', 'hidden', 'incomplete_bus_schedule')
    list_filter = ('incomplete_bus_schedule', 'majority', 't_type', 'station_type', 'suburban_zone',)
    search_fields = ('title', 'title_en', 'code_set__code')
    filter_horizontal = ('properties',)
    ordering = ('-id',)
    readonly_fields = ('type_choices', 'tablo_state_prev', 'modified_at')

    def full_path_with_links(self, station):
        return self.station_full_path(station)

    full_path_with_links.allow_tags = True
    full_path_with_links.short_description = _(u"Путь к станции")

    @classmethod
    def station_full_path(cls, station, include_station=False):
        parts = filter(None, [
            station,
            station.settlement,
            station.region,
            station.country
        ])

        if not include_station:
            parts = parts[1:]

        def make_url(object, bold=False):
            if bold:
                return u'<a href="%s" ' \
                       u'style="font-size: 110%%; font-weight: bold;">%s</a>' % \
                       (get_rasp_admin_url(object, relative_only=True), object.title)
            return u'<a href="%s">%s</a>' % (get_rasp_admin_url(object, relative_only=True),
                                             object.title)

        if not parts:
            return u""

        parts = [make_url(parts[0], True)] + map(make_url, parts[1:])
        return u" &lt; ".join(parts)

    def change_view(self, request, object_id, form_url='', extra_context=None):
        import locale

        extra_context = extra_context or {}
        extra_context['MODEL_LANGUAGES'] = {}
        for lang in settings.MODEL_LANGUAGES:
            extra_context['MODEL_LANGUAGES'][lang] = locale.normalize(lang).split('.')[0].replace('_', '-')

        return super(StationAdmin, self).change_view(request, object_id, form_url, extra_context=extra_context)


admin.site.register(Station, StationAdmin)


class PointSynonymAdmin(RaspExportModelAdmin):
    def has_delete_permission(self, request, obj=None):
        return False

    def has_add_permission(self, request):
        return False

    list_display = ('title', 'point', 'language')
    search_fields = ('title',)

    def formfield_for_foreignkey(self, db_field, request, **kwargs):
        existing_content_types = set(PointSynonym.objects.values_list('content_type_id', flat=True))

        if db_field.name == "content_type":
            kwargs["queryset"] = ContentType.objects.filter(id__in=existing_content_types).order_by('app_label', 'model')

        return super(PointSynonymAdmin, self).formfield_for_foreignkey(db_field, request, **kwargs)


admin.site.register(PointSynonym, PointSynonymAdmin)


class Station2SettlementAdmin(RaspExportModelAdmin):
    raw_id_fields = ('station', 'settlement')
    fieldsets = (
        ('Profile field', {
            'fields': ('station', 'settlement')
        }),
    )
    list_display = ('station', 'settlement')
    search_fields = ('station__title', 'settlement__title')


admin.site.register(Station2Settlement, Station2SettlementAdmin)


class CompanyTariffFilter(admin.SimpleListFilter):
    title = _(u'Багажные тарифы')
    parameter_name = 'baggage_tariff'

    def lookups(self, request, model_admin):
        return [
            ('no_baggage_tariff', _(u'Нет багажного тарифа')),
            ('with_baggage_tariff', _(u'Есть багажный тариф'))
        ]

    def queryset(self, request, queryset):
        if self.value() == 'no_baggage_tariff':
            return queryset.exclude(id__in=self.company_tariffs())
        elif self.value() == 'with_baggage_tariff':
            return queryset.filter(id__in=self.company_tariffs())
        return queryset

    @staticmethod
    def company_tariffs():
        return CompanyTariff.objects.values_list('avia_company')


class CompanySynonymInline(admin.TabularInline):
    model = CompanySynonym
    extra = 0


class CompanyAdmin(TranslationsMixin, RaspExportModelAdmin):
    # Для того чтобы в админке появился филд в зависимости от условий нужно
    # переопределить 2 метода get_form и get_fieldsets

    def get_form(self, request, obj=None, **kwargs):
        if obj and obj.t_type_id == TransportType.PLANE_ID:
            fields = flatten_fieldsets(self.get_fieldsets(request))
            fields.append('alliance')
            fields += L_field.admin_fields(Company, ['bonus_name'])
            kwargs['fields'] = fields

        return super(CompanyAdmin, self).get_form(request, obj, **kwargs)

    def get_fieldsets(self, request, obj=None):
        fieldsets = super(CompanyAdmin, self).get_fieldsets(request, obj)

        if obj and obj.t_type_id == TransportType.PLANE_ID:
            fieldsets = deepcopy(fieldsets)
            # Чтобы было красиво филд нужно добавить в третий fieldset
            fieldsets[2][1]['fields'] += ('alliance',)
            # Название программы для часто летающих пассажиров добавляем в конец
            fieldsets += ((_(u'название программы для часто летающих пассажиров'), {
                'classes': ('collapse',),
                'fields': L_field.admin_fields(Company, ['bonus_name'])
            }),)

        return fieldsets

    def has_delete_permission(self, request, obj=None):
        return False

    def has_add_permission(self, request):
        return False

    raw_id_fields = ('home_station', 'country')
    fieldsets = (
        (_(u'Локальные названия'), {
            'fields': [
                'title',
                'short_title'
            ]
        }),
        (_(u'Переводы'), {
            'fields': L_field.admin_fields(Company, ['title', 'short_title']),
            'classes': ['collapse']
        }),
        (_(u'Новые переводы'), {
            'fields': ['new_L_title', 'new_L_short_title']
        }),
        (_(u'Коды'), {
            'fields': ('sirena_id', 'iata', 'icao', 'icao_ru')
        }),
        (_(u'Лого'), {
            'fields': ('logo', 'icon', 'logo_mono', 'logo_bgcolor', 'logo_complex', ('svg_logo2', 'svg2png_logo2'),)
        }),
        (_(u'SEO'), {
            'fields': ('seo_description_approved', 'seo_description_i18n')
        }),
        ('Profile field', {
            'fields': ('address', 't_type', 'country',
                       'home_station', 'url', 'priority', 'email',
                       'contact_info', 'phone', 'phone_booking', 'description', 'hidden', 'strange', 'is_freight',
                       'meta_title', 'meta_description')
        }),
        (_(u'Контактные данные для регистрации'), {
            'fields': [
                'registration_url',
                'registration_phone',
            ]
        }),
        (_(u'Переводы контактных данных для регистрации'), {
            'fields': L_field.admin_fields(Company, [
                'registration_url',
                'registration_phone',
            ])
        }),
    )
    list_display = ('title', 'short_title', 'title_en', 'slug', 'address', 'country',
                    'home_station', 'sirena_id', 'iata', 'icao', 'url')
    list_filter = ('t_type', CompanyTariffFilter)
    search_fields = ('title', 'title_en', 'address', 'url', 'iata',
                     'sirena_id', 'icao', 'contact_info', 'phone', 'phone_booking', 'description')
    readonly_fields = ('logo_complex', 'seo_description_i18n')
    inlines = (AviaCompanyInline, CompanySynonymInline)

    def logo_complex(self, obj):
        if obj.logo_mono:
            return '<img src="{}" style="background-color: {}"/>'.format(obj.logo_mono.url, obj.logo_bgcolor or '#000000')
        else:
            return ''

    logo_complex.allow_tags = True
    logo_complex.short_description = _(u'Предпросмотр составного логотипа')


admin.site.register(Company, CompanyAdmin)


class CompanySynonymAdmin(RaspExportModelAdmin):
    raw_id_fields = ('company', )
    list_display = ('synonym', 'language', 'company')
    search_fields = ['synonym', 'company__title']


admin.site.register(CompanySynonym, CompanySynonymAdmin)


class AllianceCompanyInline(admin.TabularInline):
    """ Нужен для отображения партнёров внутри альянса """

    model = Company
    extra = 0
    can_delete = False
    fields = ('title', 'iata', 'hidden')
    readonly_fields = ('title', 'iata', 'hidden')

    def has_add_permission(self, request):
        return False


class AviaAllianceAdmin(TranslationsMixin, RaspExportModelAdmin):
    def has_delete_permission(self, request, obj=None):
        return False

    def has_add_permission(self, request):
        return False

    list_display = ('title', 'enabled')
    list_filter = ('enabled',)
    fieldsets = (
        (_(u'Основное'), {'fields': [
            'title',
            'description',
            'logo_svg',
            'enabled',
        ]}),
        (_(u'Переводы'), {
            'fields': L_field.admin_fields(AviaAlliance, ['title', 'description']),
            'classes': ['collapse']
        }),
        (_(u'Новые переводы'), {'fields': ['new_L_title', 'new_L_description']}),
    )
    inlines = [AllianceCompanyInline]


admin.site.register(AviaAlliance, AviaAllianceAdmin)


class TransportSchemeAdmin(RaspExportModelAdmin):
    raw_id_fields = ('transport',)
    fieldsets = (
        ('Profile field', {
            'fields': ('img', 'title', 'company', 'transport', 'transport_img')
        }),
    )
    list_display = ('img', 'title', 'company', 'transport', 'transport_img')
    search_fields = ('title', 'company')


admin.site.register(TransportScheme, TransportSchemeAdmin)


class RouteAdmin(RaspExportModelAdmin):
    def mark_for_recount(self, request, queryset):
        RThread.objects.filter(route__in=queryset).update(changed=True)
        messages.add_message(request, messages.SUCCESS, _(u"Маршруты помечены как измененные"))
    mark_for_recount.short_description = "Mark as changed"

    actions = ['mark_for_recount']

    fieldsets = (
        (_(u'Последнее обновление'), {'fields': ('modified_at',)}),
        (_(u'Параметры'), {
            'fields': (
                'route_uid',
                'route_number',
                'route_title',
                't_type',
                'script_protected',
                'comment',
                'hidden',
                'style',
                'red_metaroute_link'
            )
        }),
    )
    list_display = ('route_uid', 'route_number', 'route_title', 't_type', 'supplier')
    list_display_links = ('route_uid',)
    list_filter = ('t_type', CommentListFilter, 'supplier')
    search_fields = ('route_uid', 'rthread__title', 'rthread__number')
    readonly_fields = ('red_metaroute_link', 'route_uid', 'route_title', 'route_number', 'modified_at')

    def red_metaroute_link(self, obj):
        if not obj.red_metaroute:
            return ''
        return u'<a href="%s" target="_blank">%s: %s</a>' % (
            obj.red_metaroute.redadmin_link,
            obj.red_metaroute.id,
            obj.red_metaroute.full_name
        )
    red_metaroute_link.short_description = _(u'Рейс красного админа')
    red_metaroute_link.allow_tags = True

    def route_title(self, obj):
        return obj.route_title
    route_title.short_description = _(u'Название')

    def route_number(self, obj):
        return obj.route_number
    route_number.short_description = _(u'Номер')


admin.site.register(Route, RouteAdmin)


class RThreadTypeAdmin(RaspExportModelAdmin):
    list_display = ('title', 'code')


admin.site.register(RThreadType, RThreadTypeAdmin)


class RThreadAdminForm(forms.ModelForm):
    class Meta:
        model = RThread
        exclude = []

    def clean_template_code(self):
        self.cleaned_data['template_code'] = (self.cleaned_data['template_code'] or u'').strip()

        if self.cleaned_data['template_code']:
            from travel.avia.admin.scripts.schedule.utils.afmasktext import AFTextBuilder, AFTextBuildError, AFTextMatchError

            try:
                AFTextBuilder().build_range_day_texts(
                    self.cleaned_data['template_code'].split(u'#')[-1]
                )
            except (AFTextBuildError, AFTextMatchError) as e:
                raise forms.ValidationError(
                    _(u'Ошибка разбора шабона: "{template}": {error}').format(
                        template=self.cleaned_data['template_code'], error=unicode(e)
                    )
                )

        return self.cleaned_data['template_code'].strip()

    def clean_t_subtype(self):
        t_subtype = self.cleaned_data['t_subtype']
        t_type = self.cleaned_data['t_type']
        if t_subtype and t_type != t_subtype.t_type:
            raise forms.ValidationError(
                _(u'Неверный подтип транспорта "{}" для типа транспорта "{}"').format(t_subtype, t_type))

        return t_subtype


class RThreadAdmin(OrderingOnlyOnSmallQuerysetModelAdminMixin, RaspExportModelAdmin):
    form = RThreadAdminForm
    raw_id_fields = ('route', 'company', 't_model', 'basic_thread')
    readonly_fields = ('translated_manual_days_texts', 'pseudo_data', 'modified_at')
    fieldsets = (
        (_(u'Последнее обновление'), {'fields': ('modified_at',)}),
        (_(u'Названия'), {
            'fields': (
                'title',
                'title_short',
                'title_tr',
                'title_uk',
                'is_manual_title'
            )
        }),
        (_(u'Основные параметры'), {
            'fields': (
                'route',
                't_type',
                't_subtype',
                'number',
                'hidden_number',
                'ordinal_number',
                'old_thread_number',
                'uid',
                'type',
                'tariff_type',
                'company',
                't_model',
                'basic_thread',
                'express_type',
                'express_lite',
                'schedule_plan',
                'supplier',
                'changed',
                'path_and_time_unchanged',
                'is_circular',
                'is_combined',
                'hidden',
            )
        }),
        (_(u'Время и дни хождения'), {
            'fields': (
                'tz_start_time',
                'time_zone',
                'begin_time',
                'end_time',
                'period_int',
                'density',
                'comment',
                'pseudo_data',
                'year_days'
            )
        }),
        (_(u'Тексты дней хождений'), {
            'fields': ('template_code', 'template_start', 'template_end', 'template_text')
        })
    )
    list_display = ('number', 'title', 'time_zone', 'tz_start_time', 'get_type', 'get_model', 'uid')
    list_display_links = ('number', 'title', 'uid')
    list_filter = ('type', 't_type', CommentListFilter, 'supplier', 'schedule_plan')
    search_fields = ('number', 'uid', 'title')
    ordering = ('number', 'title')

    def restore_changes(self, request, queryset):
        basic = RThreadType.objects.get(code='basic')
        for thread in queryset:
            if thread.type == basic:
                thread_mask = RunMask(thread.year_days)
                for date_ in thread.get_change_date_list():
                    thread_mask[date_] = True
                for change in thread.thread_changes.all():
                    change.delete()
                thread.year_days = str(thread_mask)
                thread.changed = True
                thread.save()
        messages.add_message(request, messages.SUCCESS, _(u"Изменения и отмены ниток были восстановлены."))

    restore_changes.short_description = _(u"Восстановить изменения")

    def mark_as_changed(self, request, queryset):
        queryset.update(changed=True)
        messages.add_message(request, messages.SUCCESS, _(u"Нитки помечены как измененные"))

    mark_as_changed.short_description = _(u"Пометить как изменённые")

    actions = ['restore_changes', 'mark_as_changed']

    def extrapolate_and_calc_cysix_template_texts(self, thread):
        return

    def save_model(self, request, obj, form, change):
        from travel.avia.admin.scripts.schedule.utils import afmasktext

        if obj.id:
            old_thread = RThread.objects.get(pk=obj.id)
            if old_thread.year_days != obj.year_days:
                self.extrapolate_and_calc_cysix_template_texts(obj)

        obj.template_code = obj.template_code.strip()

        obj.translated_manual_days_texts = u''

        if obj.template_code:
            if u'#' in obj.template_code:
                tz, obj.template_code = obj.template_code.split(u'#')
                tz = tz or 'local'

                texts = afmasktext.shift_and_build_days_texts(obj, template_timezone=tz)
            else:
                texts = afmasktext.shift_and_build_days_texts(obj)

            if texts:
                obj.translated_manual_days_texts = json.dumps(texts)

        obj.save()

        # RASP-3317
        for station_id in request.POST.getlist('station'):
            station_admin = StationAdmin(Station, self.admin_site)
            station = Station._default_manager.get(id=station_id)

            approx = request.POST['station_%s_approx' % station_id]

            if not approx:
                try:
                    lat = float(request.POST['station_%s_lat' % station_id])
                    lng = float(request.POST['station_%s_lng' % station_id])
                except ValueError:
                    continue

                old_lat = station.latitude
                old_lng = station.longitude

                if old_lat != lat or old_lng != lng:
                    station.latitude = lat
                    station.longitude = lng
                    message = _(u"Сменили координаты при редактировании из нитки "
                                u"%(uid)s, c %(old_lat)s %(old_lng)s на %(lat)s %(lng)s") % \
                               {'uid': obj.uid, 'old_lat': old_lat, 'old_lng': old_lng, 'lat': lat, 'lng': lng}

                    station.save()
                    station_admin.log_change(request, station, message)

    def fill_coords(self, s0, s1, unknown):
        def not_none(*args):
            for arg in args:
                if arg is not None:
                    return arg

        lng0 = s0.station.longitude
        lat0 = s0.station.latitude
        t0 = not_none(s0.departure, s0.arrival)

        lng1 = s1.station.longitude
        lat1 = s1.station.latitude
        t1 = not_none(s1.departure, s1.arrival)

        if t1 == t0:
            klng = lng1
            klat = lat1
        else:
            klng = (lng1 - lng0) / (t1 - t0)
            klat = (lat1 - lat0) / (t1 - t0)

        for rtstation in unknown:
            t = not_none(rtstation.departure, rtstation.arrival)

            rtstation.station.longitude = lng0 + klng * (t - t0)
            rtstation.station.latitude = lat0 + klat * (t - t0)
            rtstation.station.approx = True

    def get_interpolated_stations(self, rtstations):
        railway_interpolated_stations = {}

        filepath = os.path.join(settings.LOG_PATH, 'railway_interpolated_stations.txt')
        if os.path.exists(filepath):
            f = open(filepath)
            for line in f:
                parts = line.replace('\n', '').split('\t')
                railway_interpolated_stations[int(parts[0])] = ', '.join(parts[1:])

        interpolated_stations = {}
        for rtstation in rtstations:
            if rtstation.station_id in railway_interpolated_stations:
                interpolated_stations[rtstation.station_id] = railway_interpolated_stations[rtstation.station_id]

        return interpolated_stations

    def get_far_stations(self, rtstations):
        railway_far_stations = {}
        filepath = os.path.join(settings.LOG_PATH, 'railway_far_stations.txt')
        if os.path.exists(filepath):
            f = open(filepath)
            for line in f:
                parts = line.replace('\n', '').split('\t')
                railway_far_stations[int(parts[0])] = ', '.join(parts[1:])

        far_stations = {}
        for rtstation in rtstations:
            if rtstation.station_id in railway_far_stations:
                far_stations[rtstation.station_id] = railway_far_stations[rtstation.station_id]

    def fill_arrival_and_departure(self, rtstations, naive_start_dt):
        if rtstations[0].tz_departure is None:
            return rtstations

        start_dt = rtstations[0].get_departure_loc_dt(naive_start_dt)

        for rtstation in rtstations:
            arr_dt = rtstation.get_arrival_loc_dt(naive_start_dt)
            arrival = (arr_dt - start_dt).total_seconds() / 60 if arr_dt else None

            dep_dt = rtstation.get_departure_loc_dt(naive_start_dt)
            departure = (dep_dt - start_dt).total_seconds() / 60 if dep_dt else None

            rtstation.arrival, rtstation.departure = arrival, departure

        return rtstations

    def change_view(self, request, object_id, form_url='', extra_context=None):
        today = environment.today()
        thread = get_object_or_404(RThread, id=object_id)
        naive_start_dt = datetime.combine(today, thread.tz_start_time)

        rtstations = list(thread.rtstation_set.select_related('station'))

        preprev = None
        prev = None
        unknown = []

        has_thread_tariffs = ThreadTariff.objects.filter(thread_uid=thread.uid).exists()

        rtstations = self.fill_arrival_and_departure(rtstations, naive_start_dt)

        for rtstation in rtstations:
            if rtstation.station.latitude and (rtstation.arrival is not None
                                               or rtstation.departure is not None):
                if prev:
                    self.fill_coords(prev, rtstation, unknown)
                    preprev = prev
                    unknown = []

                prev = rtstation
            else:
                unknown.append(rtstation)

        if unknown and preprev and prev:
            self.fill_coords(preprev, prev, unknown)

        json_fields = []
        for json_field in ['translated_manual_days_texts',
                           'translated_days_texts',
                           'translated_except_texts']:

            value = getattr(thread, json_field)
            if value:
                try:
                    value = json.loads(value)
                except ValueError:
                    pass

            json_fields.append({
                'name': json_field,
                'verbose_name': RThread._meta.get_field(json_field).verbose_name,
                'value': json.dumps(value, ensure_ascii=False, encoding='utf8', indent=2)
            })

        return super(RThreadAdmin, self).change_view(request, object_id, extra_context={
            'rtstations': rtstations,
            'interpolated_stations': self.get_interpolated_stations(rtstations),
            'far_stations': self.get_far_stations(rtstations),
            'json_fields': json_fields,
            'has_thread_tariffs': has_thread_tariffs,
        })


admin.site.register(RThread, RThreadAdmin)


class RTStationAdmin(RaspExportModelAdmin):
    raw_id_fields = ('thread', 'station', 'arrival_t_model', 'departure_t_model', 'terminal')
    fieldsets = (
        (_(u'Смешанные поля'), {
            'fields': (
                'thread',
                'station',
                'is_technical_stop',
                'terminal',
                'arrival_t_model',
                'departure_t_model',
                'is_virtual_end',
                'is_combined',
                'platform'
            )
        }),
        (_(u'Новая информация об остановке'), {
            'fields': ('time_zone', 'tz_arrival', 'tz_departure')
        }),
        (_(u"Нечеткие признаки"), {
            'fields': ('is_fuzzy', 'is_searchable_from', 'is_searchable_to',
                       'in_station_schedule', 'in_thread')
        }),
        (_(u"Информация о направлениях"), {
            'fields': ('arrival_direction',
                       'arrival_subdir',
                       'departure_direction',
                       'departure_subdir',
                       'is_from_subdir')
        }),
        (u"Code sharing", {
            'fields': ('arrival_code_sharing',
                       'departure_code_sharing')
        }),
    )
    list_display = ('get_thread_number', 'get_station_title', 'terminal_letter', 'time_zone', 'tz_arrival',
                    'tz_departure', 'is_technical_stop', 'get_directions', 'arrival_subdir', 'departure_subdir',
                    'get_is_fuzzy', 'get_is_combined')
    list_filter = ()
    search_fields = ('thread__number',)

    def get_is_fuzzy(self, rtstation):
        return rtstation.is_fuzzy

    get_is_fuzzy.allow_tags = True
    get_is_fuzzy.short_description = u"fuzzy"
    get_is_fuzzy.boolean = True

    def get_is_combined(self, rtstation):
        return rtstation.is_combined

    get_is_combined.allow_tags = True
    get_is_combined.short_description = _(u"Согл. Перес.")
    get_is_combined.boolean = True

    def get_station_title(self, rtstation):
        return StationAdmin.station_full_path(rtstation.station, include_station=True)

    get_station_title.allow_tags = True
    get_station_title.short_description = _(u"Станция")

    def terminal_letter(self, rtstation):
        if rtstation.terminal_id:
            return rtstation.terminal.name
    terminal_letter.short_description = u"term"

    def get_thread_number(self, rtstation):
        return rtstation.thread.number

    get_thread_number.short_description = _(u"Нитка")

    def get_directions(self, rtstation):
        if not rtstation.arrival_direction and not rtstation.departure_direction:
            return None
        if rtstation.arrival_direction != rtstation.departure_direction:
            directions = filter(None, [rtstation.arrival_direction and rtstation.arrival_direction.title,
                                       rtstation.departure_direction and rtstation.departure_direction.title])
            return u" - ".join(directions)
        else:
            return rtstation.arrival_direction.title

    get_directions.short_description = _(u"Направление")


admin.site.register(RTStation, RTStationAdmin)


class RoutePathAdmin(RaspExportModelAdmin):
    raw_id_fields = ('station_from', 'station_to')
    list_display = ('station_from', 'station_to', 'status_direct', 'status_back')


admin.site.register(RoutePath, RoutePathAdmin)


class PlatformTranslationAdmin(TranslationsMixin, RaspExportModelAdmin):
    list_display = ('__unicode__', ) + \
        tuple(L_field.admin_fields(PlatformTranslation, ['platform'], show_local_field=False))

    fieldsets = (
        (_(u'Смешанные поля'), {
            'fields': ('description',),
        }),
        (_(u'Платформа'), {
            'fields': L_field.admin_fields(PlatformTranslation, ['platform'], show_local_field=True),
            'classes': ('collapse')
        }),
        (_(u'Новые переводы'), {
            'fields': ['new_L_platform']
        }),
    )


admin.site.register(PlatformTranslation, PlatformTranslationAdmin)


class StationPhoneAdmin(RaspExportModelAdmin):
    raw_id_fields = ('station',)
    fieldsets = (
        ('Profile field', {
            'fields': ('station', 'phone', 'note')
        }),
    )
    list_display = ('station', 'phone', 'note')
    search_fields = ('phone', 'station__title', 'note')


admin.site.register(StationPhone, StationPhoneAdmin)


class Route2CompanyAdmin(RaspExportModelAdmin):
    raw_id_fields = ('company',)
    fieldsets = (
        ('Profile field', {
            'fields': ('number', 'company')
        }),
    )
    list_display = ('number', 'company')
    search_fields = ('number', 'company__title')


admin.site.register(Route2Company, Route2CompanyAdmin)


class SupplierAdmin(RaspExportModelAdmin):
    fieldsets = (
        ('Profile field', {'fields': (
            'title', 'code', 'filename', 'hide_in_filters',
            'add_to_route_changes_report', 'exclude_from_external_api',
            'logo',
        )}),
        (_(u'Продажи'), {'fields': (
            'can_buy_ru', 'can_buy_ua', 'can_buy_tr',
            'sale_url_template', 'sale_start_days', 'sale_stop_hours',
        )}),
    )
    list_display = ('id', 'title', 'code', 'can_buy_ru', 'can_buy_ua', 'can_buy_tr')
    search_fields = ('title', 'code')
    ordering = ('id',)


admin.site.register(Supplier, SupplierAdmin)


class StationCodeAdmin(RaspExportModelAdmin):
    fieldsets = (
        ('Profile field', {
            'fields': ('station', 'code', 'system')
        }),
    )
    raw_id_fields = ('station',)
    list_filter = ('system',)
    search_fields = ('station__title', 'code')
    list_display = ('station', 'code', 'system')


admin.site.register(StationCode, StationCodeAdmin)


class SettlementCodeAdmin(RaspExportModelAdmin):
    fieldsets = (
        ('Profile field', {
            'fields': ('settlement', 'code', 'system')
        }),
    )
    raw_id_fields = ('settlement',)
    list_filter = ('system',)
    search_fields = ('settlement__title', 'code')
    list_display = ('settlement', 'code', 'system')


admin.site.register(SettlementCode, SettlementCodeAdmin)


class HolidayAdmin(RaspExportModelAdmin):

    readonly_fields = ('name_tanker_key',)

    fieldsets = (
        ('Holiday field', {
            'fields': (
                'title', 'first_segment_first_day', 'first_segment_last_day',
                'second_segment_first_day', 'second_segment_last_day',
                'is_active', 'name_tanker_key'
            )
        }),
    )

    list_display = (
        'title', 'first_segment_first_day', 'first_segment_last_day',
        'second_segment_first_day', 'second_segment_last_day',
        'is_active'
    )


admin.site.register(Holiday, HolidayAdmin)


class PreHolidayAdmin(RaspExportModelAdmin):
    fieldsets = (
        ('Profile field', {
            'fields': ('date', 'title')
        }),
    )
    list_display = ('date', 'title')


admin.site.register(PreHoliday, PreHolidayAdmin)


class SuggestAdmin(RaspExportModelAdmin):
    list_filter = ('type', 'ttype')
    search_fields = ('title',)
    list_display = ('title', 'ttype', 'object_id', 'type',
                    'full_title_ru', 'full_title_tr')
    raw_id_fields = ('region', 'country')


admin.site.register(Suggest, SuggestAdmin)


class DirectionMarkerInline(admin.TabularInline):
    model = DirectionMarker
    raw_id_fields = ('station', 'direction', 'prev', 'next')
    extra = 0


class DirectionAdmin(RaspExportModelAdmin):
    list_filter = ('suburban_zone', )
    search_fields = ('title', 'code')
    list_display = ('title', 'title_from', 'title_to', 'code')
    inlines = (DirectionMarkerInline,)


admin.site.register(Direction, DirectionAdmin)


class TranslateAdmin(RaspExportModelAdmin):
    search_fields = ('value',)
    list_display = ('value',)


admin.site.register(DirectionTranslate, TranslateAdmin)
admin.site.register(DirectionFromTranslate, TranslateAdmin)


class DirectionMarkerAdmin(RaspExportModelAdmin):
    raw_id_fields = ('station', 'direction', 'prev', 'next')
    list_filter = ('direction',)
    list_display = ('order', 'direction_name', '__unicode__',
                    'title_from', 'title_to',
                    'arrow_type')

    @transaction.atomic
    def shift_down(self, request, queryset):
        directions = set([dm.direction for dm in queryset])
        if len(directions) > 1:
            messages.add_message(request, messages.ERROR, _(u"Нужно выбирать маркеры из одного направления"))
        else:
            direction = list(directions)[0]
            # Начиная с минимального номера увеличиваем всем порядок на 1
            min_order = queryset.order_by('order')[0].order
            for dm in direction.directionmarker_set.filter(order__gte=min_order).order_by('-order'):
                dm.order += 1
                dm.save()

        messages.add_message(request, messages.SUCCESS, _(u"Порядковые номера маркеров были увеличены"))
        return HttpResponseRedirect("./?direction=%d" % direction.id)

    shift_down.short_description = _(u"Сдвинуть вниз")

    actions = ['shift_down']


admin.site.register(DirectionMarker, DirectionMarkerAdmin)


class SuburbanZoneAdmin(TranslationsMixin, RaspExportModelAdmin):
    raw_id_fields = ('settlement',)
    list_display = ('title', 'code')
    ordering = ('title',)
    fieldsets = (
        (None, {'fields': ['code', 'settlement', 'title_from', 'title_to', 'title']}),
        (_(u'Переводы'), {
            'fields': L_field.admin_fields(SuburbanZone, ['title']),
            'classes': ('collapse',)
        }),
        (_(u'Новые переводы'), {'fields': ['new_L_title']}),
    )

    def get_urls(self):
        from django.conf.urls import url

        def wrap(view):
            def wrapper(*args, **kwargs):
                return self.admin_site.admin_view(view)(*args, **kwargs)
            return update_wrapper(wrapper, view)

        urlpatterns = super(SuburbanZoneAdmin, self).get_urls()

        info = self.model._meta.app_label, self.model._meta.model_name

        urlpatterns = [
            url(r'^(.+)/notices/$', wrap(self.notices), name='%s_%s_notices' % info),
        ] + urlpatterns

        return urlpatterns

    def notices(self, request, object_id):
        from travel.avia.admin.suburban_change_notices.models import Notice

        current_date = environment.today()
        if request.GET.get('date'):
            current_date = datetime.strptime(request.GET.get('date'), '%Y-%m-%d').date()

        zone = SuburbanZone.objects.get(id=int(object_id))

        filter_ = Q(start_date__gte=current_date) | \
            Q(start_date__lt=current_date, end_date__gte=current_date)

        notices = Notice.objects \
                        .filter(filter_, directions__suburban_zone=zone) \
                        .distinct() \
                        .order_by('start_date', 'end_date')

        json_notices = []

        for notice in notices:
            json_notices.append({
                'title': notice.title,
                'text': notice.text,
                'mobile_text': notice.mobile_text,
                'start_date': unicode(notice.start_date),
                'end_date': unicode(notice.end_date),
                'type': notice.type,
                'type_display': unicode(notice.get_type_display()),
                'directions': [{'code': d.code, 'title': d.title} for d in notice.directions.all()]
            })

        return HttpResponse(json.dumps({'data': json_notices}, ensure_ascii=False, indent=2), content_type='application/json; charset=utf-8')


admin.site.register(SuburbanZone, SuburbanZoneAdmin)


class TicketOfficeAdmin(RaspExportModelAdmin):
    raw_id_fields = ('settlement',)
    search_fields = ('settlement__title', 'title', 'address', 'postal_code')
    list_display = ('title', 'settlement', 'plane', 'train', 'postal_code', 'address', 'home_number')
    fieldsets = (
        ('Profile field', {
            'fields': ('title', 'settlement', 'postal_code', 'address',
                       'home_number', 'address2', 'phone', 'fax',
                       'home_site', 'email', 'plane',
                       'train', 'companies')
        }),
        (_(u"Координаты"), {
            'classes': ('ymaps-point',),
            'fields': ('longitude', 'latitude'),
        })
    )


admin.site.register(TicketOffice, TicketOfficeAdmin)


class ExternalDirectionMarkerAdmin(RaspExportModelAdmin):
    list_filter = ('external_direction',)
    list_display = ('order', 'direction_name', '__unicode__')
    raw_id_fields = ('station', 'external_direction')

    @transaction.atomic
    def shift_down(self, request, queryset):
        external_directions = set([edm.external_direction for edm in queryset])
        if len(external_directions) > 1:
            messages.add_message(request, messages.ERROR, _(u"Нужно выбирать маркеры из одного направления"))
        else:
            external_direction = list(external_directions)[0]
            # Начиная с минимального номера увеличиваем всем порядок на 1
            min_order = queryset.order_by('order')[0].order
            for edm in external_direction.externaldirectionmarker_set.filter(order__gte=min_order).order_by('-order'):
                edm.order += 1
                edm.save()

        messages.add_message(request, messages.SUCCESS, _(u"Порядковые номера маркеров были увеличены"))
        return HttpResponseRedirect("./?external_direction=%d" % external_direction.id)

    shift_down.short_description = _(u"Сдвинуть вниз")

    actions = ['shift_down']


admin.site.register(ExternalDirectionMarker, ExternalDirectionMarkerAdmin)


class ExternalDirectionMarkerInline(admin.TabularInline):
    model = ExternalDirectionMarker
    raw_id_fields = ('station', )
    extra = 0


class ExternalDirectionAdmin(RaspExportModelAdmin):
    search_fields = ('title', 'code')
    list_filter = ('suburban_zone', )
    list_display = ('title', 'code')
    inlines = (ExternalDirectionMarkerInline,)
    raw_id_fields = ('base_station', )


admin.site.register(ExternalDirection, ExternalDirectionAdmin)


class SuburbanTariffAdmin(RaspExportModelAdmin):
    list_display = ('code', 'tariff')


admin.site.register(SuburbanTariff, SuburbanTariffAdmin)


class RouteImportInfoAdmin(RaspExportModelAdmin):
    list_display = ('number', 'supplier', 't_type')
    fieldsets = (
        ('Identification fields', {
            'fields': ('number', 'supplier', 'route_uid')
        }),
        ('Add. info fields', {
            'fields': ('t_type',)
        }),
    )


admin.site.register(RouteImportInfo, RouteImportInfoAdmin)


class PageAdmin(RaspExportModelAdmin):
    list_display = ('title', 'code')


admin.site.register(Page, PageAdmin)


class TeaserCountryInline(admin.TabularInline):
    model = Teaser.countries.through
    raw_id_fields = ('country',)
    extra = 1


class TeaserSettlementInline(admin.TabularInline):
    model = Teaser.settlements.through
    raw_id_fields = ('settlement',)
    extra = 1


class TeaserStationInline(admin.TabularInline):
    model = Teaser.stations.through
    raw_id_fields = ('station',)
    extra = 1


class TeaserDirectionInline(admin.TabularInline):
    model = Teaser.directions.through
    raw_id_fields = ('externaldirection',)
    extra = 1


class TeaserThreadInline(admin.TabularInline):
    model = Teaser.threads.through
    raw_id_fields = ('rthread',)
    extra = 1


class TeaserCompanyInline(admin.TabularInline):
    model = Teaser.companies.through
    raw_id_fields = ('company',)
    extra = 1


class TeaserPageInline(admin.TabularInline):
    model = Teaser.pages.through
    raw_id_fields = ('page',)
    extra = 1


class TeaserAdmin(RaspExportModelAdmin):
    fieldsets = (
        (
            None,
            {'fields': ('title', 'content', 'mobile_content',
                        'url', 'image', 'template',
                        'mode', 'importance',
                        'date_start', 'date_finish')}
        ),
        (
            _(u'Где показывать'),
            {'fields': ('lang', 'national_version',
                        'is_active_rasp', 'is_active_ticket', 'is_active_export')}
        ),
    )

    list_display = ('title', 'content', 'lang', 'national_version',
                    'is_active_rasp', 'is_active_ticket', 'is_active_export',
                    'mode', 'importance',
                    'date_start', 'date_finish')
    list_filter = ('is_active_rasp', 'is_active_ticket', 'is_active_export',
                   'importance', 'mode',
                   'lang', 'national_version')
    inlines = [TeaserPageInline, TeaserCountryInline, TeaserSettlementInline, TeaserStationInline,
               TeaserDirectionInline, TeaserThreadInline, TeaserCompanyInline]
    ordering = ('-is_active_rasp',)

    def queryset(self, request):
        return Teaser.base_objects.all()


admin.site.register(Teaser, TeaserAdmin)


class DeLuxeTrainAdmin(TranslationsMixin, RaspExportModelAdmin):
    list_display = ('title_admin', 'numbers', 'deluxe', 'high_speed')
    search_fields = ('numbers', 'title')
    list_filter = ('deluxe', 'high_speed')

    fieldsets = (
        (None, {
            'fields': (
                'numbers',
                'deluxe',
                'high_speed',
            )
        }),
        (_(u'Название'), {
            'fields': L_field.admin_fields(DeLuxeTrain, ['title'], show_local_field=True),
            'classes': ['collapse']
        }),
        (_(u'Новые переводы'), {'fields': ['new_L_title']})
    )

    def title_admin(self, obj):
        return obj.L_title_short() or _(u'название не указано')


admin.site.register(DeLuxeTrain, DeLuxeTrainAdmin)


class AeroexTariffAdmin(RaspExportModelAdmin):
    list_filter = ('type', 'precalc')
    list_display = ('station_from', 'station_to', 'reverse', 'type', 'tariff', 'currency', 'suburban_search', 'precalc')
    raw_id_fields = ('station_from', 'station_to')
    search_fields = ('station_from__title', 'station_to__title')

    def get_changelist(self, request, **kwargs):
        """
        Returns the ChangeList class for use on the changelist page.
        """
        from travel.avia.admin.lib.admin_options import ChangeListNoDefaultOrdering
        return ChangeListNoDefaultOrdering


admin.site.register(AeroexTariff, AeroexTariffAdmin)


class NotTariffTrainAdmin(RaspExportModelAdmin):
    list_display = ('number',)


admin.site.register(NotTariffTrain, NotTariffTrainAdmin)

admin.site.register(ExpressTypeLite)


class ExpressNumberAdmin(RaspExportModelAdmin):
    list_display = ('number', 'supplier', 'express_type')


admin.site.register(ExpressNumber, ExpressNumberAdmin)


class RedirectAdmin(RaspExportModelAdmin):
    list_display = ('old_url', 'new_url')
    search_fields = ('old_url', 'new_url')


admin.site.register(Redirect, RedirectAdmin)


class StationTerminalAdmin(RaspExportModelAdmin):
    raw_id_fields = ('station',)


admin.site.register(StationTerminal, StationTerminalAdmin)


class ThreadTariffAdmin(RaspExportModelAdmin):
    raw_id_fields = ('station_from', 'station_to')
    list_display = ('thread_uid', 'station_from', 'station_to', 'tariff')
    search_fields = ('thread_uid', 'station_from__title', 'station_to__title')

    def get_changelist(self, request, **kwargs):
        """
        Returns the ChangeList class for use on the changelist page.
        """
        from travel.avia.admin.lib.admin_options import ChangeListNoDefaultOrdering
        return ChangeListNoDefaultOrdering


admin.site.register(ThreadTariff, ThreadTariffAdmin)


class TariffTypeAdmin(RaspExportModelAdmin):
    list_display = ('title', 'category', 'code', 'order')
    list_filter = ['category', 'tariff_groups__title']
    filter_horizontal = ['tariff_groups']


admin.site.register(TariffType, TariffTypeAdmin)


@admin.register(TariffGroup)
class TariffGroupAdmin(RaspExportModelAdmin):
    list_display = ['title']


class TrainSchedulePlanAdmin(RaspExportModelAdmin):
    list_display = ('title', 'start_date', 'end_date')
    fieldsets = (
        ('', {
            'fields': ('title', 'code', 'start_date', 'end_date', 'appendix_type')
        }),
    )


admin.site.register(TrainSchedulePlan, TrainSchedulePlanAdmin)


class DefaultPointAdmin(RaspExportModelAdmin):
    list_display = ('title', 'settlement', 'station')
    raw_id_fields = ('station', 'settlement')


admin.site.register(DefaultPoint, DefaultPointAdmin)


class ReplaceExceptionAdmin(RaspExportModelAdmin):
    list_display = ('city_from', 'city_to', 'station_to')
    raw_id_fields = ('city_from', 'city_to', 'station_to')


admin.site.register(ReplaceException, ReplaceExceptionAdmin)


class ThreadScheduleAdmin(RaspExportModelAdmin):
    list_display = ('thread', 'start_time', 'end_time', 'week_days', 'frequency', 'times')
    raw_id_fields = ('thread',)
    search_fields = ('thread__route__number', 'thread__number')


admin.site.register(ThreadSchedule, ThreadScheduleAdmin)


class StationPassageAdmin(RaspExportModelAdmin):
    list_display = ('station_from', 'terminal_from', 'station_to', 'terminal_to', 'duration')
    raw_id_fields = ('station_from', 'terminal_from', 'station_to', 'terminal_to')


admin.site.register(StationPassage, StationPassageAdmin)

admin.site.register(CodeSystem)


class TrainPseudoStationMapAdmin(RaspExportModelAdmin):
    search_fields = ('number', 'pseudo_station__title', 'station__title')
    list_display = ('number', 'pseudo_station', 'station')
    raw_id_fields = ('station', 'pseudo_station')


admin.site.register(TrainPseudoStationMap, TrainPseudoStationMapAdmin)


class DistrictAdmin(RaspExportModelAdmin):
    search_fields = ('title', 'settlement__title', 'region__title')
    list_display = ('title', 'settlement', 'region', 'country')
    raw_id_fields = ('region', 'settlement')

    def country(self, obj):
        return (
            obj.settlement and obj.settlement.country
        ) or (
            obj.region and obj.region.country
        )

    country.short_description = _(u'страна')


admin.site.register(District, DistrictAdmin)


class PseudoRegionCountriesInline(admin.TabularInline):
    model = PseudoRegion.countries.through
    raw_id_fields = ('country',)
    extra = 1


class PseudoRegionAdmin(RaspExportModelAdmin):
    list_display = ('title',)
    fields = ('title',)
    inlines = [PseudoRegionCountriesInline]


admin.site.register(PseudoRegion, PseudoRegionAdmin)


class RegionWhiteListForm(ModelForm):
    regions = forms.ModelMultipleChoiceField(Region.objects.all(),
                                             label=_(u'Области'),
                                             widget=FilteredSelectMultiple(_(u'Области'),
                                                                           False,
                                                                           attrs={'rows': '10'}))

    class Meta:
        model = RegionWhiteList
        exclude = []


class RegionWhiteListAdmin(RaspExportModelAdmin):
    form = RegionWhiteListForm


admin.site.register(RegionWhiteList, RegionWhiteListAdmin)


class StationExpressAliasAdmin(RaspExportModelAdmin):
    list_display= ('station', 'alias')
    raw_id_fields = ('station',)


admin.site.register(StationExpressAlias, StationExpressAliasAdmin)


class GortransCityLinkAdmin(RaspExportModelAdmin):
    list_display = ('city', 'url_a')
    raw_id_fields = ('city',)


admin.site.register(GortransCityLink, GortransCityLinkAdmin)


class StationTeleportMapAdmin(RaspExportModelAdmin):
    list_display = ('station_from', 'settlement_from', 'station_to', 'settlement_to', 'teleport_time')
    raw_id_fields = ('station_from', 'settlement_from', 'station_to', 'settlement_to')


admin.site.register(StationTeleportMap, StationTeleportMapAdmin)


class SettingsAdmin(RaspExportModelAdmin):
    list_display = ('name', 'code', 'value', 'help_text')
    readonly_fields = ('type', 'code')

    def has_delete_permission(self, request, obj=None):
        return False

    def has_add_permission(self, request):
        return False


admin.site.register(Setting, SettingsAdmin)


class FaceDirectionAdmin(RaspExportModelAdmin):
    list_display = ('departure_settlement', 'arrival_settlement', 'ttype')
    list_filter = ('ttype',)
    raw_id_fields = ('arrival_settlement', 'departure_settlement')


admin.site.register(FaceDirection, FaceDirectionAdmin)


class CustomSpecialOfferSettlementFromFilter(SimpleListFilter):
    title = _(u'Город отправления')
    parameter_name = 'settlement_from__id__exact'

    def lookups(self, request, model_admin):
        settlement_from_ids = SpecialOffer.objects.values_list("settlement_from", flat=True).distinct()
        settlements_from = Settlement.objects.in_bulk(settlement_from_ids).items()

        return settlements_from

    def queryset(self, request, queryset):
        if self.value():
            return queryset.filter(settlement_from=self.value())

        else:
            return queryset


class SpecialOfferAdmin(RaspExportModelAdmin):
    def company_icon_img(self, obj):
        if obj.company.icon:
            content = '<img src="%s" width="16" height="16" title="%s" alt="%s"/>' % (obj.company.icon.url, obj.company.L_title(), obj.company.iata)
        else:
            content = obj.company.iata

        return '<a href="/admin/www/company/%s/" target="_blank">%s</a>' % (obj.company.id, content)

    company_icon_img.allow_tags = True
    company_icon_img.short_description = _(u'А/к')

    list_display = (
        'settlement_from', 'settlement_to', 'company_icon_img', 'tariff',
        'date_start', 'date_end',
        # 'period_start', 'period_end',
        # 'reverse', 'online_booking'
    )
    raw_id_fields = ('settlement_from', 'settlement_to', 'airport_from', 'airport_to', 'company')
    search_fields = (
        'company__iata',
        'settlement_from__iata',
        'settlement_to__iata',
        'settlement_to__title_ru',
    )
    list_filter = (
        'reverse', 'service_class', 'online_booking',
        CustomSpecialOfferSettlementFromFilter,
    )


admin.site.register(SpecialOffer, SpecialOfferAdmin)


class TransportSubtypeAdmin(RaspExportModelAdmin):
    list_filter = ('t_type',)
    list_display = ('code', 'title_ru', 't_type', 'color')


admin.site.register(TransportSubtype, TransportSubtypeAdmin)


class TransportSubtypeColorAdmin(RaspExportModelAdmin):
    pass


admin.site.register(TransportSubtypeColor, TransportSubtypeColorAdmin)


class DefaultSuggestAdmin(RaspExportModelAdmin):
    fieldsets = (
        ('', {
            'fields': ('suggest_type_id', 'national_version', 'priority')
        }),
        ('Значения', {
            'fields': L_field.admin_fields(DefaultSuggest, ['value'], show_local_field=True)
        }),
    )
    list_display = ('suggest_type_id', 'national_version', 'value', 'priority')


admin.site.register(DefaultSuggest, DefaultSuggestAdmin)


class CompanyOfficeAdmin(RaspExportModelAdmin):
    search_fields = ('title',
                     'settlement__country__title_ru', 'settlement__country__title_en',
                     'settlement__country__title_uk', 'settlement__country__title_tr',
                     'settlement__title_ru', 'settlement__title_en',
                     'settlement__title_tr', 'settlement__title_uk',
                     'company__title_ru', 'company__title_tr',
                     'company__title_en', 'company__title_uk')

    list_display = ('title', 'country', 'settlement', 'main_station')

    raw_id_fields = ('company', 'settlement', 'main_station')
    list_filter = ('is_main',)

    fieldsets = (
        ('', {
            'fields': ('title', 'company', 'settlement', 'main_station', 'address', 'contact_info',
                       'phone', 'phone_booking', 'description', 'is_main')
        }),
        (_(u'Координаты'), {
            'classes': ('ymaps-point',),
            'fields': ('longitude', 'latitude'),
        }),
    )

    def country(self, obj):
        return unicode(obj.settlement.country)


admin.site.register(CompanyOffice, CompanyOfficeAdmin)

admin.site.register(Script, ScriptAdmin)
admin.site.register(ScriptResult, ScriptResultAdmin)


class ServiceAdmin(RaspExportModelAdmin):
    list_display = ('code', 'description', 'team')


class TeamAdmin(RaspExportModelAdmin):
    list_display = ('code', 'description')


admin.site.register(Service, ServiceAdmin)
admin.site.register(Team, TeamAdmin)
