# coding=utf-8
import os
import subprocess

from django.conf import settings
from django.contrib import admin
from django.core.urlresolvers import reverse
from django.http import HttpResponse
from django.shortcuts import redirect
from django.utils.html import format_html
from django.conf.urls import url
from django.utils.translation import ugettext_lazy as _

from travel.avia.library.python.common.models.scripts import ScriptResult, Script

from travel.avia.admin.lib.admin_options import RaspExportModelAdmin


class ScriptResultInline(admin.TabularInline):
    model = ScriptResult
    verbose_name = "Script Result"
    verbose_name_plural = "Script Results"

    ordering = ['-started_at']
    exclude = ('log_file',)

    readonly_fields = (
        'script',
        'success',
        'started_at',
        'finished_at',
        'get_log_action',
        'go_to_result_action'
    )

    def get_log_action(self, obj):
        styles = 'width: 100%;display: inline-block;text-align: center;box-sizing: border-box;'
        return format_html(
            '<a class="button" style="{}" href="{}">Показать логи</a>',
            styles,
            reverse('admin:result-get-log', args=[obj.pk]),
        )
    get_log_action.short_description = 'Запустить'

    def go_to_result_action(self, obj):
        styles = 'width: 100%;display: inline-block;text-align: center;box-sizing: border-box;'
        return format_html(
            '<a class="button" style="{}" href="{}">Перейти</a>',
            styles,
            reverse('admin:www_scriptresult_change', args=[obj.pk]),
        )
    go_to_result_action.short_description = 'Перейти'

    # права
    def has_delete_permission(self, request, obj=None):
        return False

    def has_add_permission(self, request):
        return False


class ScriptAdmin(RaspExportModelAdmin):
    inlines = [ScriptResultInline]
    list_display = (
        'code',
        'enabled_in_development',
        'enabled_in_testing',
        'enabled_in_stress',
        'enabled_in_production',
        'run_stats',
        'last_error_run',
        'last_success_run',
        'is_success',
        'run_script_action',
    )
    ordering = ('code',)

    fieldsets = (
        (_(u'Общее'), {'fields': ('code', 'description', 'timeout', 'command')}),
        (_(u'Флаги'), {
            'fields': (
                (
                    'enabled_in_development',
                    'enabled_in_testing',
                    'enabled_in_stress',
                    'enabled_in_production'
                ),
            )
        }),
        (_(u'Статистика'), {
            'fields': (
                ('run_stats',),

            )
        }),

        (_(u'Действия'), {
            'fields': (
                ('run_script_action_in_main', 'last_success_run', 'last_error_run'),
            )
        }),
    )

    readonly_fields = ('run_stats', 'last_success_run', 'last_error_run', 'run_script_action_in_main')

    # динамические поля
    def is_success(self, obj):
        result = ScriptResult.objects.filter(
            script=obj
        ).order_by('-started_at').first()

        status = '?'
        color = 'blue'

        if result and result.finished_at:
            if result.success:
                status = '+'
                color = 'green'
            else:
                status = '-'
                color = 'red'

        return format_html(
            '<b style="color: {}">{}</b>',
            color,
            status,
        )
    is_success.short_description = 'Статус'

    def run_script_action_in_main(self, obj):
        styles = 'display: inline-block;box-sizing: border-box;'
        return format_html(
            '<a class="button" style="{}" href="{}">Запустить</a>',
            styles,
            reverse('admin:run-script', args=[obj.pk]),
        )
    run_script_action_in_main.short_description = 'Запустить'

    def run_script_action(self, obj):
        styles = 'width: 100%;display: inline-block;text-align: center;box-sizing: border-box;'
        return format_html(
            '<a class="button" style="{}" href="{}">Запустить</a>',
            styles,
            reverse('admin:run-script', args=[obj.pk]),
        )
    run_script_action.short_description = 'Запустить'

    def run_stats(self, obj):
        results = list(ScriptResult.objects.filter(
            script=obj
        ))

        count_success = len([r for r in results if r.success and r.finished_at])
        count_errors = len([r for r in results if not r.success and r.finished_at])
        count_pending = len([r for r in results if not r.success and r.finished_at is None])

        return format_html(''.join('<span style="color: {}"> {}</span>'.format(color, count) for color, count in [
            ('green', count_success),
            ('red', count_errors),
            ('blue', count_pending),
        ]))
    run_stats.short_description = 'S/F/P'

    def last_error_run(self, obj):
        script_result = ScriptResult.objects.filter(
            script=obj,
            success=False,
            finished_at__isnull=False
        ).order_by('-started_at').first()

        if not script_result:
            return '-'

        return format_html(
            '<a href="{}">{}</a>',
            reverse('admin:www_scriptresult_change', args=[script_result.id]),
            script_result.started_at.strftime('%Y-%m-%d %H:%M')
        )
    last_error_run.short_description = 'Последняя ошибка'

    def last_success_run(self, obj):
        script_result = ScriptResult.objects.filter(
            script=obj,
            success=True,
            finished_at__isnull=False
        ).order_by('-started_at').first()

        if not script_result:
            return '-'

        return format_html(
            '<a href="{}">{}</a>',
            reverse('admin:www_scriptresult_change', args=[script_result.id]),
            script_result.started_at.strftime('%Y-%m-%d %H:%M')
        )
    last_success_run.short_description = 'Последний успех'

    # ручки
    def process_run_script(self, request, script_id, *args, **kwargs):
        script = Script.objects.get(id=script_id)

        command = 'python {root}/manage.py run_task --code {code}'.format(
            root=settings.PROJECT_PATH,
            code=script.code
        )

        subprocess.Popen(command, shell=True)

        return redirect(reverse('admin:www_script_change', args=[script.pk]))

    def get_urls(self):
        urls = super(ScriptAdmin, self).get_urls()
        custom_urls = [
            url(
                r'^(?P<script_id>.+)/run_script/$',
                self.admin_site.admin_view(self.process_run_script),
                name='run-script',
            )
        ]
        return custom_urls + urls


class ScriptResultAdmin(RaspExportModelAdmin):
    list_display = (
        'id',
        'script',
        'success',
        'started_at',
        'finished_at',
        'get_log_action'
    )

    readonly_fields = (
        'script',
        'success',
        'started_at',
        'finished_at',
        'log_file',
        'get_log_action'
    )

    # динамические поля
    def get_log_action(self, obj):
        styles = 'width: 100%;display: inline-block;text-align: center;box-sizing: border-box;'
        return format_html(
            '<a class="button" style="{}" href="{}">Показать логи</a>',
            styles,
            reverse('admin:result-get-log', args=[obj.pk]),
        )
    get_log_action.short_description = 'Запустить'

    # ручки
    def process_get_log(self, request, script_result_id, *args, **kwargs):
        result = ScriptResult.objects.get(id=script_result_id)

        log_file_path = result.log_file
        if not log_file_path:
            return HttpResponse(u'Нет логов - нет проблем', content_type='text/plain; charset=utf-8')
        if not os.path.isfile(log_file_path):
            return HttpResponse(u'Нет логов - нет проблем', content_type='text/plain; charset=utf-8')

        with open(log_file_path) as log_file:
            logs = log_file.read()

        return HttpResponse(logs, content_type='text/plain; charset=utf-8')

    def get_urls(self):
        urls = super(ScriptResultAdmin, self).get_urls()
        custom_urls = [
            url(
                r'^(?P<script_result_id>.+)/process_get_log/$',
                self.admin_site.admin_view(self.process_get_log),
                name='result-get-log',
            )
        ]
        return custom_urls + urls

    # права
    def has_delete_permission(self, request, obj=None):
        return False

    def has_add_permission(self, request):
        return False
