# -*- coding: utf-8 -*-
from __future__ import absolute_import, division, unicode_literals

import bisect
import csv
import decimal
import logging
import math
import os
import tempfile
import time
from collections import defaultdict
from datetime import datetime, timedelta
from itertools import count, islice, takewhile

import ujson
from django.conf import settings
from django.core.management.base import BaseCommand
from django.db.models import Sum
from django.template.loader import render_to_string
from django.utils.html import escape

from travel.avia.library.python.avia_data.models.balance import BalancePriceListRedirectPrepared
from travel.avia.library.python.common.models.partner import PartnerEmail, Partner
from travel.avia.library.python.common.lib.mail import SmailikClient
from travel.avia.admin.lib.logs import add_stdout_handler, create_current_file_run_log
from travel.avia.admin.lib.partner_errors_collector import PartnerErrorsCollector

from ._yt_send_partners_stat import calculate_stat_by_partners

log = logging.getLogger(__name__)
create_current_file_run_log()
add_stdout_handler(log)


def get_cost_text_by_nv(nv, cost):
    if int(cost) == cost:
        cost = int(cost)

    if nv == 'ru':
        return '{} RUB'.format(cost)

    if nv == 'com':
        return '{} EUR'.format(cost)

    if nv == 'kz':
        return '{} KZT'.format(cost)

    if nv == 'ua':
        return '{} UAH'.format(cost)

    if nv == 'tr':
        return '{} TRY'.format(cost)

    raise ValueError('Unknown national version {}'.format(nv))


class Command(BaseCommand):
    help = 'Send to partners statistic'

    def add_arguments(self, parser):
        parser.add_argument(
            '--stdout', action='store_true', default=False,
            dest='add_stdout_handler', help='Add stdout handler',
        )

        parser.add_argument(
            '-d', '--days', type=int, default=7,
            dest='days', help='number of days to process',
        )

        parser.add_argument(
            '--partners', nargs='+',
            help='partners to send statistics',
        )

        parser.add_argument(
            '--exclude-partners', nargs='+',
            help='partners to exclude from mail list. If partners option set, this option will be ignored',
        )

    def _fetch_costs(self, begin, end, billing_client_ids):
        filters = {
            'event_date__gte': begin,
            'event_date__lte': end,
        }

        if billing_client_ids:
            filters['billing_client_id__in'] = billing_client_ids

        costs_by_patner_and_nv = defaultdict(dict)
        costs = BalancePriceListRedirectPrepared.objects.filter(
            **filters
        ).values(
            'billing_client_id',
            'national_version',
        ).annotate(
            cost=Sum('cost')
        ).values_list(
            'billing_client_id', 'national_version', 'cost',
        )

        for billing_client_id, nv, cost in costs:
            costs_by_patner_and_nv[billing_client_id][nv] = cost

        return costs_by_patner_and_nv

    def _get_partners(self, partners, exclude_partners):
        """ Return list of dicts with keys code, title, billing_client_id """
        if partners:
            partners_qs = Partner.objects.filter(code__in=partners)
        elif exclude_partners:
            partners_qs = Partner.objects.exclude(code__in=exclude_partners)
        else:
            partners_qs = Partner.objects.all()

        return partners_qs.values('code', 'title', 'billing_client_id')

    def _format_partner_error(self, error):
        request_response = ujson.loads(error.request_response)
        if isinstance(request_response, str):
            return {
                'data': request_response,
            }

        response_details = request_response.get('response_details', {})
        return {
            'error_type': error.error_type or '',
            'url': request_response.get('url', ''),
            'method': request_response.get('method', ''),
            'body': request_response.get('body', ''),
            'request_response': response_details.get('content', ''),
            'status_code': response_details.get('status_code', ''),
        }

    def _format_partner_errors(self, errors_by_partner):
        return {
            partner: [self._format_partner_error(error) for error in errors if error.request_response]
            for partner, errors in errors_by_partner.iteritems()
        }

    def get_partner_errors(self, partner_codes):
        collector = PartnerErrorsCollector(settings.TICKET_DAEMON_DNS_COMPONENT, log)
        return self._format_partner_errors(collector.collect(partner_codes))

    def handle(self, *args, **options):
        log.info('Start')

        today = datetime.now().date()
        begin = today - timedelta(days=options['days'])
        end = today - timedelta(days=1)

        partners = self._get_partners(options['partners'], options['exclude_partners'])
        costs_by_partner = self._fetch_costs(begin, end, [p['billing_client_id'] for p in partners])

        log.info('Cost by partner: %r', costs_by_partner)

        partner_info_by_code = {
            p['code']: {
                'title': p['title'],
                'cost': costs_by_partner.get(p['billing_client_id'], {}),
            } for p in partners
        }

        for pcode in ('aerfolot', 'pobeda'):
            try:
                del partner_info_by_code[pcode]['cost']
            except KeyError:
                pass

        log.info('partners: %s', ', '.join(sorted(partner_info_by_code.keys())))

        qids_count, percentile_by_partner, data = calculate_stat_by_partners(begin, end)
        partner_errors = self.get_partner_errors([p['code'] for p in partners])

        pd_stat_by_partner = {
            p: pd_dataframe_from_list_of_dicts(stat)
            for p, stat in data.items()
        }

        week = {
            p: aggregate(partner_data)
            for p, partner_data in pd_stat_by_partner.items()
            if partner_data
        }
        log.info('week: %r', week)
        week = {p: agg for p, agg in week.items() if agg['good']}
        for p in week:
            week[p]['timing_90pc'] = percentile_by_partner[p]

        for rate, p in enumerate(sorted(week.keys(), key=lambda p: week[p]['timing_90pc']), start=1):
            week[p]['timing_90pc_rate'] = rate, len(week)

        for rate, p in enumerate(sorted(week.keys(), key=lambda p: week[p]['error']), start=1):
            week[p]['error_rate'] = rate, len(week)

        for rate, p in enumerate(sorted(week.keys(), key=lambda p: week[p]['empty']), start=1):
            week[p]['empty_rate'] = rate, len(week)

        for rate, p in enumerate(sorted(week.keys(), key=lambda p: week[p]['timeout']), start=1):
            week[p]['timeout_rate'] = rate, len(week)

        # for rate, p in enumerate(sorted(week.keys(), key=lambda p: week[p]['redir_fails']), start=1):
        #     week[p]['redir_fails_rate'] = rate, len(week)

        def prepare_stat_for_csv(frame):
            frame['timing_90pc'] = [round_significant(val, 3) for val in frame['timing_90pc']]
            return frame

        yandex_search_count = qids_count
        partners_count = len(week)

        log.info('Week: %r', week)

        for p in partners:
            pcode = p['code']
            try:
                pweek = week[pcode]
                partner_info = partner_info_by_code[pcode]
                send_partner_stat(
                    pcode,
                    partner_info['title'] or pcode.capitalize(),
                    begin, end,
                    pweek,
                    prepare_stat_for_csv(pd_stat_by_partner[pcode]),
                    {
                        'yandex_search_count': yandex_search_count,
                        'partners_count': partners_count,
                        'cost_by_nv': [
                            get_cost_text_by_nv(nv, cost) for nv, cost in partner_info.get('cost', {}).items()
                        ],
                    },
                    partner_errors.get(pcode),
                )

            except Exception:
                log.exception('Error while sending stat to %s', pcode)

        log.info('Done')


def aggregate(stat):
    qs = sum(stat['queries'])
    good = sum(stat['good'])

    d = {
        'queries': qs,
        'good': good,
        'timeout': sum(stat['timeout']) * 100 / qs,
        'empty': sum(stat['empty']) * 100 / qs,
        'error': sum(stat['error']) * 100 / qs,
        # 'redir_fails': sum(stat['redir_fails']) * 100 / sum(stat['redirs']),
    }
    return d


def summ(items):
    if not items:
        return 0
    return sum(items)


def round_significant(value, count=1):
    v = value or 0
    v = decimal.Decimal(v)
    try:
        if v <= 0:
            return v
    except decimal.InvalidOperation:
        return value
    return round(v, count - 1 - int(math.floor(math.log10(v))))


def cut_and_round_significant(value, count):
    if count > 0 and 0 < value < 10 ** (-count):
        return 0

    return round_significant(value, count)


def pd_dataframe_from_list_of_dicts(data):
    columns = data[0].keys()
    series = {k: [row.get(k) for row in data] for k in columns}
    return DataFrame(series, columns=columns)


class DataFrame(dict):
    def __init__(self, series, columns):
        super(DataFrame, self).__init__(series)
        self.columns = columns

    def to_csv(self, filepath, datetime_format, index=False):
        with open(filepath, 'wb') as f:
            writer = DictWriterTypesAware(
                f,
                fieldnames=self.columns,
                encoding='utf8',
                delimiter=','.encode('utf-8'),
                datetime_format=datetime_format
            )
            writer.writeheader()
            for row in zip(*[self[field] for field in self.columns]):
                writer.writerow(dict(zip(self.columns, row)))


class DictWriterTypesAware(csv.DictWriter):
    def __init__(self, *args, **kwargs):
        self.encoding = kwargs.pop('encoding', 'utf8')
        self.errors = kwargs.pop('errors', 'strict')
        self.datetime_format = kwargs.pop('datetime_format', '%Y-%m-%d %H:%M:%S')
        csv.DictWriter.__init__(self, *args, **kwargs)

    def _dict_to_list(self, rowdict):
        for k, value in rowdict.iteritems():
            rowdict[k] = self.convert_value(value)
        return csv.DictWriter._dict_to_list(self, rowdict)

    def convert_value(self, value):
        if isinstance(value, datetime):
            value = value.strftime(self.datetime_format)
        if isinstance(value, unicode):
            return value.encode(self.encoding, self.errors)
        elif value is None:
            return ""
        else:
            return unicode(value).encode(self.encoding, self.errors)


GREEN = '#28a528'
GREY = '#666666'
RED = '#c82828'


Z = .1
for j in range(30):
    Z = 1/(1+Z)
Z4 = Z**4


RATES_POS, COLORS = zip(*sorted({
    1: GREEN,
    1-Z4: GREY,
    Z4: RED,
}.items()))


def rate_color(pos):
    num, den = pos
    return COLORS[bisect.bisect(RATES_POS, 1-num/den)]


def send_partner_stat(pcode, pname, begin, end, week, stat, ctx, errors):
    log.info('send_partner_stat: %s [%s - %s] %s', pcode, begin, end, week)

    emailtype = 'stat'

    if settings.SEND_MAIL_TO_PARTNERS:
        emails = [
            pe.email.strip()
            for pe in PartnerEmail.objects.filter(
                partner__code=pcode, emailtype__code=emailtype)
        ]
        bcc = ['avia-partnerstat-copy@yandex-team.ru']
        if not emails:
            log.warning('Email type [%s] not found for partner: %s', emailtype, pcode)
            return
    else:
        emails = ['avia-partnerstat-copy-test@yandex-team.ru']
        bcc = None

    ctx['now'] = datetime.now()
    daterange_human = format_daterange(begin, end)
    subject = 'Статистика по работе вашего API за {}'.format(daterange_human)

    ctx['daterange_human'] = daterange_human

    ctx['big_table'] = [
        ('хороших ответов', '{}%'.format(round_significant(week['good'] * 100 / week['queries'], count=3))),
        ('запросов к сервису', format_big(round_significant(week['queries'], count=3))),
        ('ответов от сервиса', format_big(round_significant(week['good'], count=3))),
    ]

    ctx['table'] = [
        # title, color, value, rate
        (
            'Время ответа по API в 90% случаев не превышает',
            '{}с'.format(cut_and_round_significant(week['timing_90pc'], 3)) if week['timing_90pc'] else '—',
            week['timing_90pc_rate'],
            rate_color(week['timing_90pc_rate'])
        ),
        (
            'Не нашлось варианта',
            '{}%'.format(cut_and_round_significant(week['empty'], count=2)),
            week['empty_rate'],
            rate_color(week['empty_rate'])
        ),
        (
            'Сервис не ответил',
            '{}%'.format(cut_and_round_significant(week['timeout'], count=2)),
            week['timeout_rate'],
            rate_color(week['timeout_rate'])
        ),
        (
            'Ошибка в формате данных',
            '{}%'.format(cut_and_round_significant(week['error'], count=2)),
            week['error_rate'],
            rate_color(week['error_rate'])
        ),
        # (
        #     'Ошибка при переадресации',
        #     RED,
        #     round_significant(week['redir_fails'], count=2),
        #     week['redir_fails_rate'], rate_color(week['redir_fails_rate'])
        # ),
    ]

    mail_content = render_to_string('partner_stat.xml', ctx)
    csv_name = 'yandex_flights_{}_{}.csv'.format(
        begin.strftime('%Y-%m-%d'), end.strftime('%Y-%m-%d')
    )

    for email in emails:
        send_partner_mail(pcode, pname, email, bcc, subject, mail_content, stat, csv_name, errors)
        time.sleep(5)


MONTH = [''] + u"""
января февраля марта апреля мая июня июля августа сентября октября ноября декабря
""".split()


def format_daterange(begin, end):
    begin, end = sorted([begin, end])
    yb, mb, db = begin.timetuple()[:3]
    ye, me, de = end.timetuple()[:3]

    y = (yb != ye) and yb
    m = (y or mb != me) and mb
    d = (m or db != de) and db

    fbegin = ' '.join(filter(None, [
        d and str(d),
        m and MONTH[m],
        y and '{} года'.format(y)
    ]))

    fend = ' '.join(filter(None, [
        str(de),
        MONTH[me],
        ye and '{} года'.format(ye)
    ]))

    return ' — '.join(filter(None, [fbegin, fend]))


def test_format_daterange():
    assert format_daterange(
        datetime(2016, 4, 28),
        datetime(2016, 5, 5)
    ) == '28 апреля — 5 мая 2016 года'

    assert format_daterange(
        datetime(2016, 5, 5),
        datetime(2015, 4, 28)
    ) == '28 апреля 2015 года — 5 мая 2016 года'

    assert format_daterange(
        datetime(2016, 5, 5),
        datetime(2016, 5, 7)
    ) == '5 — 7 мая 2016 года'


def format_big(val):
    val *= 1000
    for one in ['', ' тыс', ' млн', ' млрд']:
        val /= 1000
        if val < 1000:
            return '{}{}'.format(val, one)
    return '{}{}'.format(format_int(int(val)), one)


def test_format_big_int():
    assert format_big(34) == '34.0'
    assert format_big(34 * 10**3) == '34.0 тыс'
    assert format_big(34 * 10**6) == '34.0 млн'
    assert format_big(34 * 10**9) == '34.0 млрд'
    assert format_big(34111 * 10**9) == '34 111 млрд'


def test_cut_and_round_significant():
    assert cut_and_round_significant(0.000001, 2) == 0


def format_int(val):
    j = iter(reversed('%s' % int(val)))
    return (' '.join(takewhile(len, (''.join(islice(j, 3)) for _ in count()))))[::-1]


def send_partner_mail(pcode, pname, email, bcc, subject, content, stat, csv_name, errors):
    log.info('Send email for %s to %s', pcode, email)

    c = SmailikClient(
        service='Flights.Yandex',
        send_from='Яндекс.Авиабилеты &lt;avia-partner@support.yandex.ru&gt;',
        reply_to='Яндекс.Авиабилеты &lt;avia-info@yandex-team.ru&gt;',
        smailik_dir='/var/spool/smailik-preparer/avia-stats/'
    )

    d = tempfile.mkdtemp(prefix='avia-stat-attachments-')
    filepath = os.path.join(d, csv_name)
    log.debug('csv file: %s', filepath)
    stat.to_csv(filepath, index=False, datetime_format='%Y-%m-%d %H:%M:%S')

    error_directory = tempfile.mkdtemp(prefix='avia-stat-attachments-')
    error_filepath = os.path.join(error_directory, 'errors.json')
    log.debug('error file: %s', error_filepath)
    with open(error_filepath, 'wb') as outp:
        ujson.dump(errors, outp)

    email = '{} &lt;{}&gt;'.format(escape(pname), email)
    c.send_email(
        email,
        escape(subject),
        content,
        [
            ('text/csv', filepath),
            ('application/json', error_filepath),
        ],
        bcc=bcc
    )
