# -*- coding: utf-8 -*-
import logging
import json
import requests

from django.core.management.base import BaseCommand

from travel.avia.library.python.avia_data.models.images import SettlementBigImage
from travel.avia.library.python.common.models.partner import DohopVendor, Partner

log = logging.getLogger(__name__)

# testing
# AVATAR_READ_URL = 'https://avatars.mdst.yandex.net'
# AVATAR_WRITE_URL = 'http://avatars-int.mdst.yandex.net:13000'

# production
AVATAR_READ_URL = 'https://avatars.mds.yandex.net'
AVATAR_WRITE_URL = 'http://avatars-int.mds.yandex.net:13000'


class Command(BaseCommand):
    help = u'''
    Команда для заливки фото городов, а также иконок компаний
    и партнёров в Аватарницу.

    Если некоторые картинки не были залиты вследствие ошибки,
    их можно перезалить отдельно, указав содержимое словаря errors
    из вывода программы в значении одноимённого ключа.

    Например:
      upload_media_to_avatar --errors '{"svg":[],"png":["foo.png"]}'
    '''

    def add_arguments(self, parser):
        parser.add_argument('--errors')

    def handle(self, **options):
        png_urls, svg_urls = fetch_urls(options['errors'])
        uploads, errors = upload_images(png_urls, svg_urls)
        dump(uploads, errors)


def push_url(field, urls):
    if field:
        urls.append('https://yastatic.net/rasp/media/{}'.format(field))


def fetch_urls(errors):
    if errors:
        return fetch_urls_from_errors(errors)
    else:
        return fetch_urls_from_database()


def fetch_urls_from_errors(errors):
    data = json.loads(errors)
    return data['png'], data['svg']


def fetch_urls_from_database():
    png_urls, svg_urls = [], []
    for i in SettlementBigImage.objects.all():
        if i.url:
            png_urls.append(i.url)
    for partners in (
        Partner.objects.all(),
        DohopVendor.objects.all(),
    ):
        for p in partners:
            push_url(p.logo_svg_ru, svg_urls)
            push_url(p.logo_svg_ua, svg_urls)
            push_url(p.logo_svg_kz, svg_urls)
            push_url(p.logo_svg_tr, svg_urls)
            push_url(p.logo_svg_com, svg_urls)
            push_url(p.logo_svg2png_ru, png_urls)
            push_url(p.logo_svg2png_ua, png_urls)
            push_url(p.logo_svg2png_kz, png_urls)
            push_url(p.logo_svg2png_tr, png_urls)
            push_url(p.logo_svg2png_com, png_urls)
    return png_urls, svg_urls


def upload_images(png_urls, svg_urls):
    uploads = {}
    errors = {'svg': [], 'png': []}
    for ext, urls, upload, error_list in (
        ('png', png_urls, upload_png_image, errors['png']),
        ('svg', svg_urls, upload_svg_image, errors['svg']),
    ):
        url_count = len(urls)
        log.debug('Uploading %d %ss', url_count, ext)
        for i, url in enumerate(urls):
            if i != 0 and i % 5 == 0:
                log.debug("  %d of %d uploaded (%d errors)", i, url_count, len(error_list))
            try:
                avatar_url = upload(url)
                uploads[url] = avatar_url
            except:
                error_list.append(url)
                raise
        log.debug("%d uploaded (%d errors)", ext, len(error_list))
    return uploads, errors


def upload_png_image(url):
    return upload_image(url, 'url', lambda r: r['sizes']['orig']['path'])


def upload_svg_image(url):
    return upload_image(url, 'url-svg', lambda r: r['extra']['svg']['path'])


def upload_image(url, param_name, extract_path):
    api = AVATAR_WRITE_URL + '/put-avia'
    params = {param_name: url}
    response = requests.get(api, params=params).json()
    path = extract_path(response)
    return AVATAR_READ_URL + path


def dump(uploads, errors):
    print(json.dumps({'uploads': uploads, 'errors': errors}))
