# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import csv
import os
import logging

from django.db import migrations

from travel.avia.admin.lib.logs import add_stdout_handler, create_current_file_run_log


def import_translations(apps, schema_editor):
    log = logging.getLogger("www.migrations")
    create_current_file_run_log()
    add_stdout_handler(log)

    import_de_station_translations(apps, log)
    import_de_settlement_translations(apps, log)


def unicode_csv_reader(utf8_data, dialect=csv.excel, **kwargs):
    csv_reader = csv.reader(utf8_data, dialect=dialect, **kwargs)
    for row in csv_reader:
        yield [unicode(cell, 'utf-8') for cell in row]


def import_de_station_translations(apps, log):
    filename = os.path.join(
        os.path.dirname(__file__), 'data/stations_de.csv'
    )

    Station = apps.get_model('www', 'Station')

    with open(filename, 'rU') as csvfile:
        csv_reader = unicode_csv_reader(csvfile, delimiter=str(';'))
        next(csv_reader, None)  # skip header
        for row in csv_reader:
            station_id, hidden, title_en, title_ru, title_de = row

            try:
                s = Station.objects.get(id=station_id)
            except Station.DoesNotExist:
                log.warning("Station DoesNotExist: %s", station_id)
                continue

            l_title = s.new_L_title

            old_de_nominative = l_title.de_nominative
            l_title.de_nominative = title_de
            l_title.save()

            log.info("Station %s updated from %s to %s", station_id, old_de_nominative, title_de)


def import_de_settlement_translations(apps, log):
    filename = os.path.join(
        os.path.dirname(__file__), 'data/settlements_de.csv'
    )

    Settlement = apps.get_model('www', 'Settlement')

    with open(filename, 'rU') as csvfile:
        csv_reader = unicode_csv_reader(csvfile, delimiter=str(';'))
        next(csv_reader, None)  # skip header
        for row in csv_reader:
            title_en, title_ru, title_de = row

            try:
                s = Settlement.objects.get(new_L_title__en_nominative=title_en, new_L_title__ru_nominative=title_ru)

            except Settlement.MultipleObjectsReturned:
                log.warning("Settlement MultipleObjectsReturned: en: %s, ru: %s", title_en, title_ru)
                continue
            except Settlement.DoesNotExist:
                log.warning("Settlement DoesNotExist: en: %s, ru: %s", title_en, title_ru)
                continue

            l_title = s.new_L_title

            old_de_nominative = l_title.de_nominative
            l_title.de_nominative = title_de
            l_title.save()

            log.info("Settlement %s updated from %s to %s", s.id, old_de_nominative, title_de)


class Migration(migrations.Migration):

    dependencies = [
        ('www', '0010_company_slug'),
    ]

    operations = [
        migrations.RunPython(import_translations, reverse_code=migrations.RunPython.noop),
    ]
