# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import csv
import os
import logging

from django.db import migrations

from travel.avia.admin.lib.logs import add_stdout_handler, create_current_file_run_log


def unicode_csv_reader(utf8_data, dialect=csv.excel, **kwargs):
    csv_reader = csv.reader(utf8_data, dialect=dialect, **kwargs)
    for row in csv_reader:
        yield [unicode(cell, 'utf-8') for cell in row]


def get_settlement_with_one_iata(settlement_model, station_model, title_en, title_ru):
    settlements_with_airport = []
    settlements = settlement_model.objects.filter(new_L_title__en_nominative=title_en, new_L_title__ru_nominative=title_ru)

    if len(settlements) == 1:
        return settlements[0]

    for s in settlements:
        stations = station_model.objects.filter(settlement_id=s.id)
        airports = [st for st in stations if st.t_type.code == 'plane']

        if len(airports) > 0:
            settlements_with_airport.append(s)

    if len(settlements_with_airport) == 0:
        raise CustomDoesNotExist

    if len(settlements_with_airport) > 1:
        raise CustomMultipleObjectsReturned

    return settlements_with_airport[0]


def import_settlement_with_one_iata(apps, schema_editor):
    log = logging.getLogger("www.migrations")
    create_current_file_run_log()
    add_stdout_handler(log)

    filename = os.path.join(
        os.path.dirname(__file__), 'data/settlements_de.csv'
    )

    settlement_model = apps.get_model('www', 'Settlement')
    station_model = apps.get_model('www', 'Station')

    with open(filename, 'rU') as csvfile:
        csv_reader = unicode_csv_reader(csvfile, delimiter=str(';'))
        next(csv_reader, None)  # skip header
        for row in csv_reader:
            title_en, title_ru, title_de = row

            try:
                s = get_settlement_with_one_iata(settlement_model, station_model, title_en, title_ru)

            except CustomDoesNotExist:
                log.warning("Settlement's CustomDoesNotExist: en: %s, ru: %s", title_en, title_ru)
                continue

            except CustomMultipleObjectsReturned:
                log.warning("Settlement CustomMultipleObjectsReturned: en: %s, ru: %s", title_en, title_ru)
                continue

            l_title = s.new_L_title
            old_de_nominative = l_title.de_nominative

            if old_de_nominative == title_de:
                log.info("Skip %s: en: %s, ru: %s (old: %s == new: %s)", s.id, title_en, title_ru, old_de_nominative, title_de)
                continue

            l_title.de_nominative = title_de
            l_title.save()

            log.info("Settlement %s updated from %s to %s", s.id, old_de_nominative, title_de)


class Migration(migrations.Migration):

    dependencies = [
        ('www', '0011_de_translations'),
    ]

    operations = [
        migrations.RunPython(import_settlement_with_one_iata, reverse_code=migrations.RunPython.noop),
    ]


class CustomMultipleObjectsReturned(Exception):
    pass


class CustomDoesNotExist(Exception):
    pass
