# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from collections import namedtuple

from django.db import migrations


FkInfo = namedtuple('FkInfo', 'referrer_table, referrer_field, is_nullable')

FK_INFOS = [
    FkInfo('avia_avgprice', 'currency_id', False),
    FkInfo('avia_bestoffers', 'currency_id', True),
    FkInfo('importinfo_trianglebusimportpackage', 'currency_id', True),
    FkInfo('importinfo_twostagebusimportpackage', 'currency_id', True),
    FkInfo('order_partnerreview', 'currency_id', True),
    FkInfo('order_queryblacklist', 'currency_id', True),
    FkInfo('www_country', 'currency_id', True),
    FkInfo('www_mincountryprice', 'currency_id', True),
    FkInfo('www_minprice', 'currency_id', True),
]


def build_create_fk_sql(fk_info):
    if fk_info.is_nullable:
        fix_values_template = """
            UPDATE {x.referrer_table} SET {x.referrer_field}=NULL
            WHERE {x.referrer_field} NOT IN (SELECT id FROM currency_currency);
        """
    else:
        fix_values_template = """
            DELETE FROM {x.referrer_table}
            WHERE {x.referrer_field} NOT IN (SELECT id FROM currency_currency);
        """
    sql_template = fix_values_template + """
        ALTER TABLE {x.referrer_table}
        ADD CONSTRAINT FOREIGN KEY ({x.referrer_field}) REFERENCES currency_currency(id)
        ON UPDATE CASCADE ON DELETE CASCADE;
    """
    return sql_template.format(x=fk_info)


create_drop_existing_fk_procedure = """
    CREATE PROCEDURE currency_drop_existing_fk(referrer_table CHAR(64), referrer_table_fk CHAR(64))
    BEGIN

        DECLARE referrer_table_constraint char(64) DEFAULT NULL;

        DECLARE done TINYINT DEFAULT FALSE;

        DECLARE referrer_table_fetcher
            CURSOR FOR
            SELECT CONSTRAINT_NAME
            FROM information_schema.KEY_COLUMN_USAGE
            WHERE REFERENCED_TABLE_NAME = "currency_currency"
              AND REFERENCED_COLUMN_NAME = "id"
              AND TABLE_NAME = referrer_table
              AND COLUMN_NAME = referrer_table_fk
              AND CONSTRAINT_SCHEMA = DATABASE();

        # catch exception when cursour runs out of rows
        DECLARE CONTINUE HANDLER FOR NOT FOUND SET done = TRUE;

        OPEN referrer_table_fetcher;

        offset_loop:
        LOOP

            FETCH NEXT FROM referrer_table_fetcher INTO referrer_table_constraint;
            IF done THEN
                LEAVE offset_loop;
            ELSE
                # prepared statements are used as a way to pass things like table name to query.
                SET @query = CONCAT('ALTER TABLE ', referrer_table, ' DROP FOREIGN KEY ', referrer_table_constraint, ';');
                PREPARE statement FROM @query;
                EXECUTE statement;
                DEALLOCATE PREPARE statement;
            END IF;

        END LOOP;
    END;
"""


class Migration(migrations.Migration):

    dependencies = [
        ('www', '0041_sync_id_for_www_region'),
    ]

    operations = [
        migrations.RunSQL('DROP PROCEDURE IF EXISTS currency_drop_existing_fk;', reverse_sql=''),
        migrations.RunSQL(create_drop_existing_fk_procedure, reverse_sql=''),
    ] + [
        migrations.RunSQL('CALL currency_drop_existing_fk("{x.referrer_table}", "{x.referrer_field}");'.format(x=fk_info), reverse_sql='')
        for fk_info in FK_INFOS
    ] + [
        migrations.RunSQL('DROP PROCEDURE IF EXISTS currency_drop_existing_fk;', reverse_sql=''),
    ] + [
        migrations.RunSQL(build_create_fk_sql(fk_info), reverse_sql='')
        for fk_info in FK_INFOS
    ]
