# -*- coding: utf-8 -*-

import logging
import contextlib

from django.core.files.base import ContentFile
from django.db import models
from django.conf import settings
from django.db.models.signals import pre_save, post_save
from django.dispatch import receiver
from django.utils.translation import ugettext_lazy as _

import travel.avia.library.python.route_search.models  # noqa
import travel.avia.admin.www.models.geo
import travel.avia.admin.www.models.precalc
import travel.avia.admin.www.models.schedule
import travel.avia.admin.www.models.tariffs
import travel.avia.admin.www.models.suggests
from travel.avia.library.python.common.models.geo import Station
from travel.avia.library.python.common.models.schedule import Company, make_slug
from travel.avia.library.python.common.utils.fields import TrimmedCharField
from travel.avia.library.python.image_processing import get_pilimage_content, svg2image


log = logging.getLogger(__name__)


class Redirect(models.Model):
    old_url = models.CharField(max_length=100, verbose_name=_(u"Старый адрес"),
                               unique=True)
    new_url = models.CharField(max_length=100, verbose_name=_(u"Новый адрес"))

    def get_new_url(self):
        if self.new_url.startswith('http://'):
            return self.new_url
        else:
            return "https://%s%s" % (settings.DOMAIN_NAME, self.new_url)

    class Meta:
        verbose_name = _(u"Редирект")
        verbose_name_plural = _(u"Редиректы")
        app_label = 'www'


class DefaultPoint(models.Model):
    u'Уточнения в поиске, выбираемые по умолчанию [RASP-4501]'
    title = TrimmedCharField(verbose_name=_(u"Название"), max_length=255, unique=True,
                             help_text=_(u"в нижнем регистре, 'москва'"))
    settlement = models.ForeignKey('www.Settlement', blank=True, null=True,
                                   verbose_name=_(u"Город"))
    station = models.ForeignKey('www.Station', blank=True, null=True,
                                verbose_name=_(u"Станция"))

    _objects = None

    def __unicode__(self):
        return self.title

    class Meta:
        verbose_name = _(u"Уточнение по умолчанию")
        verbose_name_plural = _(u"Уточнения по умолчанию")
        app_label = 'www'

    @classmethod
    def precache(cls):
        cls._objects = dict((dp.title.lower(), dp) for dp in cls.objects.all())

    @classmethod
    def check_is_default_point_for_title(cls, title, point):
        """Проверяет, является ли передаваемая точка дефолтной для названия"""

        if cls._objects is None:
            raise Exception("Default point must be precached")

        low = title.lower()
        return low in cls._objects and (cls._objects[low].settlement_id == point.id or cls._objects[low].station_id == point.id)


@receiver(pre_save, sender=Company)
def company_pre_save(instance, **kwargs):
    c = instance

    if not c.slug:
        try:
            c.slug = make_slug(c)
        except TypeError:
            pass

    try:
        orig = Company.objects.get(id=instance.id)

    except Company.DoesNotExist:
        return

    if c.svg_logo2 != orig.svg_logo2 or not c.svg_logo2:
        c.svg2png_logo2 = None  # noqa: F841


@receiver(post_save, sender=Company)
def company_post_save(instance, **kwargs):
    c = instance

    if c.svg_logo2 and not c.svg2png_logo2:
        try:
            with contextlib.closing(c.svg_logo2) as f:
                svg = f.read()

            img = svg2image(svg, size=(132, 132))
            img_content = get_pilimage_content(img)
            c.svg2png_logo2.save('unused_name', ContentFile(img_content))

        except Exception:
            log.exception('Saving svg2png_logo error')


@receiver(pre_save, sender=Station)
def station_change_coordinates_handler(sender, **kwargs):
    instance = kwargs['instance']

    if not instance.id:
        return

    try:
        old_value = Station.objects.get(id=instance.id)

        if not (instance.longitude == old_value.longitude and instance.latitude == old_value.latitude):
            travel.avia.admin.www.models.geo.RoutePath.set_changed(instance)

    except Station.DoesNotExist:
        pass
