# -*- coding: utf-8 -*-

import re

from django.conf import settings
from django.contrib.contenttypes.models import ContentType
from django.db import models
from django.utils import translation
from django.utils.translation import ugettext_lazy as _

from travel.avia.library.python.common.models.geo import Country, Region, Settlement, Station, PointSynonym, StationCode, StationMajority
from travel.avia.library.python.common.models.transport import TransportType
from travel.avia.library.python.common.models_utils.i18n import L_field
from travel.avia.library.python.common.utils.fields import TrimmedCharField
from travel.avia.library.python.common.utils.text import normalize


class Suggest(models.Model):
    LANGUAGES = settings.SUGGEST_LANGUAGES

    title = TrimmedCharField(verbose_name=_(u'наименование'), max_length=100,
                             null=False, blank=False)
    title_ru = TrimmedCharField(verbose_name=_(u'наименование (ru)'), max_length=100,
                                null=True, blank=True)
    title_uk = TrimmedCharField(verbose_name=_(u'наименование (uk)'), max_length=100,
                                null=True, blank=True)
    title_tr = TrimmedCharField(verbose_name=_(u'наименование (tr)'), max_length=100,
                                null=True, blank=True)
    title_en = TrimmedCharField(verbose_name=_(u'наименование (en)'), max_length=100,
                                null=True, blank=True)
    full_title = TrimmedCharField(verbose_name=_(u'отображаемое имя'), default='',
                                  max_length=100, null=False, blank=False)
    full_title_ru = TrimmedCharField(verbose_name=_(u'отображаемое имя (ru)'), default='',
                                     max_length=100, null=False, blank=False)
    full_title_tr = TrimmedCharField(verbose_name=_(u'отображаемое имя (tr)'), default='',
                                     max_length=100, null=False, blank=False)
    full_title_uk = TrimmedCharField(verbose_name=_(u'отображаемое имя (uk)'), default='',
                                     max_length=100, null=False, blank=False)
    full_title_en = TrimmedCharField(verbose_name=_(u'отображаемое имя (en)'), default='',
                                     max_length=100, null=False, blank=False)
    ttype = models.CharField(verbose_name=_(u'тип ТС'), max_length=100,
                             null=True, blank=True, default=None)
    type = models.CharField(verbose_name=_(u'тип объекта'), max_length=100,
                            null=True, blank=True, default=None)
    object_id = models.IntegerField(verbose_name=_(u"код объекта"), null=True, blank=True, default=None)
    majority = models.IntegerField(verbose_name=_(u"важность объекта"), null=True, blank=True, default=None)
    order = models.IntegerField(verbose_name=_(u"порядок"), null=False, default=100)
    region = models.ForeignKey(Region, verbose_name=_(u'область'), null=True, blank=True, default=None)
    country = models.ForeignKey(Country, verbose_name=_(u'страна'), null=True, blank=True, default=None)
    zone = models.ForeignKey('www.SuburbanZone', verbose_name=_(u'пригородная зона'), null=True, blank=True, default=None)

    def L_title(self):
        return self.L('title')

    def L_full_title(self):
        return self.L('full_title')

    def L(self, field):
        return self.linguistics(field, translation.get_language())

    def linguistics(self, field, lang):
        attname = '_'.join([field, lang])

        return getattr(self, attname) or getattr(self, field)

    _ids = None
    _titles = None
    _points = None

    class Meta:
        ordering = ('title',)
        verbose_name = _(u'саггест')
        verbose_name_plural = _(u'саггесты')
        app_label = 'www'

    def __unicode__(self):
        return self.title

    @classmethod
    def get_suggest_titles(cls, point):
        omonim_titles = [
            (lang, point.L_omonim_title(lang))
            for lang in Suggest.LANGUAGES
        ]

        full_titles = {
            'full_title_' + lang: u"%s, %s" % (omonim_title['title'], omonim_title['add'])
            for lang, omonim_title in omonim_titles
        }

        titles = {
            'title_' + lang: point.L_title(lang=lang)
            for lang in Suggest.LANGUAGES
        }

        titles.update(full_titles)

        return titles

    @classmethod
    def get_ttype_suggests(cls, ttype):
        ttype_id = TransportType.objects.get(code=ttype).pk

        stations_query = Station.objects.filter(rtstation__thread__route__t_type__id=ttype_id, hidden=False) \
                                        .exclude(settlement__isnull=False, settlement__hidden=True)
        if ttype == 'bus':
            majority = StationMajority.objects.get(code='not_in_tablo')
            stations_query = stations_query.exclude(majority__gt=majority.pk, settlement__isnull=False)

        # RASP-11412
        major = StationMajority.objects.get(code='not_in_search')
        stations_query = stations_query.exclude(majority_id=major.id)

        station_ids = list(stations_query.distinct().values_list('id', flat=True))

        stations = list(Station.objects.filter(id__in=station_ids))

        settlements = Settlement.objects.filter(hidden=False).in_bulk(list(set(s.settlement_id for s in stations if s.settlement_id)))

        related_settlements = Settlement.objects.filter(station2settlement__station__in=stations, hidden=False)

        type_points = [
            ('settlement', settlements.values()),
            ('settlement', related_settlements),
            ('station', stations)
        ]

        for type_, points in type_points:
            for point in points:
                titles = cls.get_suggest_titles(point)

                majority = point.majority_id

                order = getattr(point, 'suggest_order', 100)

                # Для аэропортов берем важность города
                if type_ == 'station' and point.t_type_id == 2 and point.settlement_id:
                    settlement = settlements.get(point.settlement_id)
                    majority = settlement.majority_id
                    order = settlement.suggest_order

                if type_ == 'station':
                    # RASP-9643 для станций создать саггесты в пригородных зонах
                    # всех внешних направлений, в которые входит станция
                    suburban_zones_ids = point.externaldirectionmarker_set.\
                        values_list('external_direction__suburban_zone_id', flat=True)
                else:
                    suburban_zones_ids = [point.suburban_zone_id]

                for lang in Suggest.LANGUAGES:
                    if suburban_zones_ids:
                        for suburban_zone_id in suburban_zones_ids:
                            d = dict(
                                object_id=point.id,
                                title=point.L_title(lang=lang),
                                type=type_,
                                majority=majority,
                                order=order,
                                region_id=point.region_id,
                                country_id=point.country_id,
                                zone_id=suburban_zone_id,
                                ttype=ttype)
                            d.update(titles)

                            yield d
                    else:
                        d = dict(
                            object_id=point.id,
                            title=point.L_title(lang=lang),
                            type=type_,
                            majority=majority,
                            order=order,
                            region_id=point.region_id,
                            country_id=point.country_id,
                            zone_id=point.suburban_zone_id,
                            ttype=ttype)
                        d.update(titles)

                        yield d

    @classmethod
    def get_airports_suggests(cls):
        airports = list(Station.objects.filter(t_type__id=2, hidden=False).select_related('settlement'))

        codes = {}

        station_codes = StationCode.objects.filter(
            system__code__in=['sirena', 'iata', 'icao'],
            station__in=airports
        ).select_related('system')

        for station_code in station_codes:
            codes.setdefault(station_code.station_id, {})[station_code.system.code] = station_code.code

        for airport in airports:
            omonim_titles = [(lang, airport.L_omonim_title(lang))
                             for lang in Suggest.LANGUAGES]

            airport_codes = codes.get(airport.id, {})

            for system in ['iata', 'icao', 'sirena']:
                try:
                    code = airport_codes[system]
                except KeyError:
                    continue

                full_titles = dict([
                    (
                        'full_title_' + lang,
                        u"%s (%s), %s" % (omonim_title['title'], code, omonim_title['add'])
                    )
                    for lang, omonim_title in omonim_titles
                ])

                titles = dict([('title_' + lang, airport.L_title(lang=lang)) for lang in Suggest.LANGUAGES])
                titles.update(full_titles)

                for lang in Suggest.LANGUAGES:
                    d = dict(
                        object_id=airport.id,
                        title=u"%s [%s]" % (airport.L_title(lang=lang), code),
                        type='station',
                        majority=airport.settlement and airport.settlement.majority_id or airport.majority_id,
                        order=airport.settlement.suggest_order if airport.settlement else 100,
                        region_id=airport.region_id,
                        country_id=airport.country_id,
                        zone_id=airport.suburban_zone_id,
                        ttype='plane')
                    d.update(titles)

                    yield d

    @classmethod
    def get_water_suggests(cls):
        ports = Station.objects.filter(t_type__in=TransportType.WATER_TTYPE_IDS, hidden=False).select_related('settlement')

        settlements = set()

        for port in ports:
            titles = cls.get_suggest_titles(port)

            if port.settlement:
                settlements.add(port.settlement)

            for lang in Suggest.LANGUAGES:
                port_l_title = port.L_title(lang=lang)

                # RASP-12430, пропускаем порты, название которых совпадает с названием города
                if port.settlement:
                    city_l_title = port.settlement.L_title(lang=lang)

                    if normalize(port_l_title) == normalize(city_l_title):
                        continue

                d = dict(
                    object_id=port.id,
                    title=port_l_title,
                    type='station',
                    majority=port.majority_id,
                    order=100,
                    region_id=port.region_id,
                    country_id=port.country_id,
                    zone_id=port.suburban_zone_id,
                    ttype='sea'
                )
                d.update(titles)
                yield d

                d['ttype'] = 'water'
                yield d

        related_settlements = Settlement.objects.filter(station2settlement__station__in=ports,
                                                        hidden=False)
        settlements.update(related_settlements)

        for settlement in settlements:
            titles = cls.get_suggest_titles(settlement)

            for lang in Suggest.LANGUAGES:
                d = dict(
                    object_id=settlement.id,
                    title=settlement.L_title(lang=lang),
                    type='settlement',
                    majority=settlement.majority_id,
                    order=settlement.suggest_order,
                    region_id=settlement.region_id,
                    country_id=settlement.country_id,
                    zone_id=settlement.suburban_zone_id,
                    ttype='sea'
                )
                d.update(titles)
                yield d

                d['ttype'] = 'water'
                yield d

    @classmethod
    def get_iata_city_plane_suggests(cls):
        settlements = Settlement.objects.filter(
            type_choices__contains='plane',
            iata__isnull=False,
            hidden=False,
        )

        # По-умолчанию английского нет, но для авиа они нужны
        languages = Suggest.LANGUAGES
        if 'en' not in languages:
            languages.append('en')

        for city in settlements:
            omonim_titles = [(lang, city.L_omonim_title(lang))
                             for lang in languages]

            full_titles = dict([
                (
                    'full_title_' + lang,
                    u"%s (%s), %s" % (omonim_title['title'], city.iata, omonim_title['add'])
                )
                for lang, omonim_title in omonim_titles
            ])

            titles = dict([('title_' + lang, city.L_title(lang=lang)) for lang in languages])
            titles.update(full_titles)
            for lang in languages:
                d = dict(
                    object_id=city.id,
                    title=u"%s [%s]" % (city.L_title(lang=lang), city.iata),
                    type='settlement',
                    majority=city.majority_id,
                    order=city.suggest_order,
                    region_id=city.region_id,
                    country_id=city.country_id,
                    zone_id=city.suburban_zone_id,
                    ttype='plane')
                d.update(titles)

                yield d

    @classmethod
    def get_plane_synonym_suggests(cls):
        def get_synonym_titles(point, title):
            omonim_titles = [
                (lang, point.L_omonim_title(lang))
                for lang in Suggest.LANGUAGES
            ]

            full_titles = {
                'full_title_' + lang: u"%s, %s" % (omonim_title['title'], omonim_title['add'])
                for lang, omonim_title in omonim_titles
            }

            titles = {
                'title_' + lang: title
                for lang in Suggest.LANGUAGES
            }

            titles.update(full_titles)

            return titles

        # Кажется что если со станциями и городами работать по-отдельности будет проще

        synonyms = PointSynonym.objects.filter(
            content_type=ContentType.objects.get_for_model(Settlement)
        ).values_list('object_id', 'title')

        settlement_synonyms = {}
        for object_id, title in synonyms:
            synonym = settlement_synonyms.setdefault(object_id, [])
            synonym.append(title)

        settlements = Settlement.objects.filter(
            hidden=False,
            type_choices__contains='plane',
            id__in=settlement_synonyms.keys()
        )

        for point in settlements:
            titles = settlement_synonyms.get(point.id)

            for title in titles:
                d = dict(
                    object_id=point.id,
                    title=title,
                    type='settlement_synonym',
                    majority=point.majority_id,
                    order=point.suggest_order,
                    region_id=point.region_id,
                    country_id=point.country_id,
                    zone_id=point.suburban_zone_id,
                    ttype='plane'
                )

                d.update(get_synonym_titles(point, title))

                yield d

        synonyms = PointSynonym.objects.filter(
            content_type=ContentType.objects.get_for_model(Station)
        ).values_list('object_id', 'title')

        station_synonyms = {}
        for object_id, title in synonyms:
            synonym = station_synonyms.setdefault(object_id, [])
            synonym.append(title)

        stations = Station.objects.filter(
            hidden=False,
            t_type__id=2,
            id__in=station_synonyms.keys()
        ).select_related('settlement')

        for point in stations:
            titles = station_synonyms.get(point.id)

            for title in titles:
                d = dict(
                    object_id=point.id,
                    title=title,
                    type='station_synonym',
                    majority=point.settlement and point.settlement.majority_id or point.majority_id,
                    order=point.settlement.suggest_order if point.settlement else 100,
                    region_id=point.region_id,
                    country_id=point.country_id,
                    zone_id=point.suburban_zone_id,
                    ttype='plane'
                )

                d.update(get_synonym_titles(point, title))

                yield d


class NameSearchIndex(models.Model):
    SEARCH_FIELDS = {
        Country: {'L_title': ['nominative']},
        Settlement: {'L_title': ['nominative']},
        Station: {'L_title': ['nominative'], 'L_popular_title': ['nominative']}
    }

    SPLIT_RE = re.compile(r'[ \-—().]+', re.U)

    STOPWORDS = set(ur'на с\о км st п c el km пос. тц'.split())

    text = models.CharField(max_length=255, null=False, db_index=True)
    exact = models.BooleanField(null=False)

    content_type = models.ForeignKey(ContentType, db_index=True)
    object_id = models.PositiveIntegerField()

    @classmethod
    def split(cls, normalized):
        return [
            word for word in cls.SPLIT_RE.split(normalized)
            if word and (word not in cls.STOPWORDS)
        ]

    @classmethod
    def get_records(cls):
        for model, fields in cls.SEARCH_FIELDS.items():
            ct = ContentType.objects.get_for_model(model)

            for s in PointSynonym.objects.filter(content_type=ct):
                yield (normalize(s.title), ct.id, s.object_id, 1)

            for point in model._default_manager.all():
                object_id = point.pk

                for lang in L_field.LANGUAGES:
                    for field, cases in fields.iteritems():
                        for case in cases:

                            text = getattr(point, field)(case=case, lang=lang)

                            if not text:
                                continue

                            normalized = normalize(text)

                            yield (normalized, ct.id, object_id, 1)

                            words = cls.split(normalized)

                            if not words:
                                continue

                            if len(words) > 1:
                                # Строим индекс с начала фразы, для поиска по начальным словам -
                                for i in xrange(len(words)):
                                    for j in xrange(i, len(words)):
                                        phrase = " ".join(words[i:j + 1])
                                        if not phrase == normalized:
                                            yield (phrase, ct.id, object_id, 0)

                            elif words[0] != normalized:
                                yield (words[0], ct.id, object_id, 1)

    class Meta:
        app_label = 'www'
