# -*- coding: utf-8 -*-
from django.conf import settings
from django.db import models
from django.utils.translation import ugettext_lazy as _

from travel.avia.library.python.common.models.translations import TranslatedTitle
from travel.avia.library.python.common.models_utils.i18n import new_L_field
from travel.avia.library.python.common.precache.manager import PrecachingManager
from travel.avia.library.python.common.utils.environment import get_request


class DefaultSuggest(models.Model):
    """
    Содержит список названий элементов (городов, станций,...) предлагаемых для выбора пользователя, если пользователь
    еще не начал уточнять поиск либо поиску не удовлетворяет ни один элемент
    """

    TRAIN_STATION = 1
    TRAIN_SETTLEMENT = 2
    PLANE_STATION = 3
    PLANE_SETTLEMENT = 4

    L_value = new_L_field(
        verbose_name=_(u'значение'),
        help_text=_(u'Предлагаемое значение'),
        max_length=100,
        add_local_field=True,
    )

    new_L_value = models.ForeignKey(TranslatedTitle, related_name='+')

    priority = models.IntegerField(verbose_name=_(u'приоритет'),
                                   help_text=_(u"Определяет порядок сортировки элементов"))

    # Удалить в 2.28
    lang = models.CharField(verbose_name=_(u'язык (deprecated)'),
                            help_text=_(u'Язык, для которого будет отображаться подсказка: ru, tr,...'),
                            max_length=10,
                            choices=settings.LANGUAGES)

    national_version = models.CharField(choices=settings.DOMAIN_TO_NATIONAL_VERSION.items(),
                                        max_length=25,
                                        default='ru',
                                        verbose_name=_(u'Национальная версия'))

    suggest_type_id = models.IntegerField(
        verbose_name=_(u'Suggest'),
        help_text=_(u'Suggest к которому относится элемент'),
        choices=[
            (TRAIN_SETTLEMENT, _(u'ж/д город')),
            (TRAIN_STATION, _(u'ж/д станция')),
            (PLANE_SETTLEMENT, _(u'авиа город')),
            (PLANE_STATION, _(u'авиа аэропорт')),
        ]
    )

    objects = PrecachingManager(
        keys=(
            'pk',
        ),
        select_related=(
            'new_L_value',
        ),
    )

    @classmethod
    def get_suggests(cls, suggest_type):
        national_version = get_request().NATIONAL_VERSION

        suggests = cls.objects.filter(suggest_type_id=suggest_type) \
            .filter(national_version=national_version) \
            .order_by('priority', 'value')

        return (s.L_value() for s in suggests)

    def get_local_language(self):
        lang, _ = settings.DOMAIN_LANGUAGE_MAP.get(self.national_version, 'ru')

        return lang

    class Meta:
        verbose_name = _(u'default suggest')
        verbose_name_plural = _(u'default suggests')
        ordering = ('suggest_type_id', 'lang', 'priority', 'value')
        app_label = 'www'
