# -*- coding: utf-8 -*-
from __future__ import absolute_import

import re
from datetime import datetime

from travel.avia.admin.www.templatetags import timezones
from travel.avia.admin.www.templatetags.common import class_attribute
from travel.avia.library.python.common.utils.date import DateTimeFormatter
from travel.avia.admin.lib.jinja import Library
from travel.avia.admin.xgettext.i18n import xgettext


jinja = Library()


def filter800suburban(segments):
    """RASP-3143, убираем 800-e электрички-дубли поездов"""

    trains800s = set()  # 800-е поезда

    number800re = re.compile(r'^(8\d\d)\D*')  # регэксп 800-ых номеров

    # Выбираем 800-ые поезда
    for segment in segments:
        if segment.t_type.code == 'train':
            m = number800re.match(segment.number)

            if m:
                number = m.group(1)
                key = (number, segment.departure, segment.station_from.id, segment.station_to.id)

                trains800s.add(key)

    new_segments = []

    # Убираем соответствующие им 800-ые электрички из списка сегментов
    for segment in segments:
        if segment.t_type.code == 'suburban':
            m = number800re.match(segment.number)

            if m:
                number = m.group(1)
                key = (number, segment.departure, segment.station_from.id, segment.station_to.id)

                if key in trains800s:
                    # Дубль - пропускаем
                    continue

        new_segments.append(segment)

    return new_segments


@jinja.callable
def trains_disclaimer(routes, routes2, form):
    """Если в списке маршрутов есть поезда, возвращает предупреждение"""
    message = u'<div class="b-static-text"><p>Обращаем ваше внимание, что расписание поездов является гарантированно точным только на промежуток до 45 дней вперед от текущей даты. ' \
              u'На более поздний срок в расписании возможны изменения.</p></div>'
    long_dates = (
        (
            form.cleaned_data['dateForward']
            and (form.cleaned_data['dateForward'] - datetime.now()).days > 45
        )
        or (
            form.cleaned_data['dateBackward']
            and (form.cleaned_data['dateBackward'] - datetime.now()).days > 45
        )
    )

    if long_dates:
        routes = routes['routes']
        if routes2:
            routes = routes + routes2['routes']
        if any(r.t_type.code == 'train' for r in routes):
            return message
    return ""


@jinja.callable
def route_strong_number(route, number=None):
    if route.t_type.code in ('train', 'plane'):
        return "<strong>%s</strong>" % (number or getattr(route, 'ticket_number', ''))
    else:
        if hasattr(route, 'ticket_number'):
            return route.ticket_number
        else:
            return route.number


@jinja.callable
def bus_route_showed_strong_number(scheduleroute):
    number = scheduleroute.displayed_number
    return number and '<strong>%s</strong>' % number or ''


@jinja.callable
def route_company(route):
    if route.company and not route.company.hidden:
        title = route.company.L_short_title() or route.company.L_title()

        if title:
            if not route.company.strange:
                return u"""<a class="b-link" href="/info/company/%s">%s</a>""" % \
                       (route.company.id, title)

            else:
                return title

    return ""


class time_with_date_ifchanged_simple(object):
    def __init__(self, column):
        self.column = column
        self.prev_date = None

    def __call__(self, dt):
        show_date = False
        if dt.date() != self.prev_date:
            self.prev_date = dt.date()
            show_date = True

        if show_date:
            return u''.join(xgettext(u'<date format="%d&nbsp;%B, %H:%M"/>',
                                     date=DateTimeFormatter(dt).L))
        else:
            return u''.join(xgettext(u'<time format="%H:%M"/>',
                                     time=DateTimeFormatter(dt).L))

jinja.callable(time_with_date_ifchanged_simple)


class time_with_date_ifchanged(object):
    def __init__(self, column, tz, tz_cities, naive_dt=False, wrapper="span", classes=[]):
        self.column = column
        self.tz = tz
        self.tz_cities = tz_cities
        self.prev_date = None
        self.prev_hidden_date = None
        self.naive_dt = naive_dt
        self.wrapper = wrapper
        self.classes = classes

    def format_dt(self, dt, hidden):
        show_date = False

        if hidden:
            # Если элемент скрыт, то нельзя изменять значение prev_date, но и логику
            # менять нельзя, поэтому используем аттрибут prev_hidden_date
            if dt.date() != self.prev_hidden_date:
                show_date = True
                self.prev_hidden_date = dt.date()
        else:
            if dt.date() != self.prev_date:
                self.prev_date = dt.date()
                # Также выставляем prev_hidden_date, чтобы логика для скрытых дат
                # не ломалась
                self.prev_hidden_date = self.prev_date

                show_date = True

        if show_date:
            return u''.join(xgettext(u'<strong><date-time format="%H:%M"/></strong>, '
                                     u'<date format="%d&nbsp;%B"/>',
                                     date=DateTimeFormatter(dt).L,
                                     strong=lambda content: [u'<strong>'] + content + [u'</strong>']))

        return u''.join(xgettext(u'<strong><time format="%H:%M"/></strong>',
                                 time=DateTimeFormatter(dt).L,
                                 strong=lambda content: [u'<strong>'] + content + [u'</strong>']))

    def __call__(self, dt, hidden=False):
        if self.naive_dt:
            return self.format_dt(dt, hidden)

        cls = class_attribute('js-datetime', 'js-no-repeat',
                              *(self.classes + [timezones.json_dt(dt, self.tz_cities, self.column)]))

        display_dt = timezones.astimezone(dt, self.tz)

        return u"""<%s%s>%s</%s>""" % \
                   (self.wrapper, cls, self.format_dt(display_dt, hidden), self.wrapper)

jinja.callable(time_with_date_ifchanged)
