# -*- coding: utf-8 -*-
from __future__ import absolute_import

import re
from django.core.urlresolvers import reverse

from ylog.utils import log_warning

from travel.avia.library.python.common.models.geo import Settlement, Station
from travel.avia.library.python.common.utils.weather import WeatherError, get_weather_airport_info, get_weather_city_info
from travel.avia.admin.www.templatetags.common import url


def one_station_title(title, filter):
    """ В зависимости от фильтра отрывает из названия первую или последнюю станцию """
    stations = title.split(' - ')
    if len(stations) < 2:
        return title
    if filter == 'departure':
        return stations[1]
    else:
        return stations[0]


def geo_data(child, parent=None):
    """Принимает гео-объект и рисует геоспуск от России до него
    Вход: объект, объект-родитель (опционально)
    Выход: список гео объектов"""
    def get_dict(o):
        if isinstance(o, Settlement):
            link = url('city', o.id)
        else:
            link = reverse('change_' + str(o.__class__.__name__).lower(), args=[o.id])

        return {'id': o.id,
                'title': isinstance(o, Station) and o.get_popular_title() or o.title,
                'url': link,
                'type': str(o.__class__.__name__).lower()
                }

    result = []
    if isinstance(child, Station):
        if parent:
            result.append(get_dict(parent))
            if parent.region and not parent.region.alone():
                result.append(get_dict(parent.region))
            if parent.country:
                result.append(get_dict(parent.country))

        elif child.settlement_id:
            result.append(get_dict(child.settlement))
            if child.settlement.region:
                if not child.settlement.region.alone():
                    result.append(get_dict(child.settlement.region))
                if child.settlement.region.country:
                    result.append(get_dict(child.settlement.region.country))
            elif child.settlement.country:
                result.append(get_dict(child.settlement.country))

        elif child.region_id and not child.region.alone():
            result.append(get_dict(child.region))
            result.append(get_dict(child.region.country))

        elif child.country:
            result.append(get_dict(child.country))

    if isinstance(child, Settlement):
        if child.region and not child.region.alone():
            result.append(get_dict(child.region))
        if child.country:
            result.append(get_dict(child.country))
        elif child.region and child.region.country:
            result.append(get_dict(child.region.country))

    result.reverse()
    if isinstance(child, Station):
        if child.settlement and child.title == child.settlement.title:
            title = child.station_type.name
        else:
            title = child.get_popular_title()
    else:
        title = child.title

    for obj in result:
        link = None

        if obj['type'] == 'settlement':
            link = url('city', obj['id'])

        yield link, obj['title']

    yield None, title


def weather_city(request, settlement):
    try:
        return get_weather_city_info(request, settlement._geo_id)

    except WeatherError:
        log_warning()
        return {}


def weather_data(request, station):
    """Погода на станции"""
    weather = {}

    try:
        if station.t_type_id == 1 and station.settlement_id:
            weather = weather_city(request, station.settlement)

        if station.t_type_id == 2:
            icao = station.get_code('icao') or station.get_code('weather_icao')
            if icao:
                weather = get_weather_airport_info(request, icao)

    except WeatherError:
        log_warning()

    return weather


def translate_wind_direction(direction):
    table = dict(zip(u'сюзв', 'nswe'))
    return re.sub(ur'[сюзв]', lambda m: table[m.group(0)], direction.lower(), re.U)
