# -*- coding: utf-8 -*-
from __future__ import absolute_import

import operator
import json
import re
import time
from datetime import datetime, date

from django.conf import settings
from django.http import QueryDict
from django.utils.safestring import mark_safe, mark_for_escaping
from django.utils.http import urlencode

from travel.avia.library.python.common.utils.date import human_date_without_year
from travel.avia.admin.lib.jinja import Library

jinja = Library()


@jinja.filter
def signed_number(number):
    """ Выводит число со знаком
        Вход: число
        Выход: число со знаком (+3, 0, -2)
    """
    if not number:
        return '0'

    result = str(number)
    if number > 0:
        result = '+'+result
    return result


@jinja.filter
def min_to_hour(minutes):
    """ Переводит минуты в часы """
    number_type = minutes / 6 % 10 == 0 and int or float
    return number_type(minutes) / 60


@jinja.filter
def nbsp(value):
    """ Заменяет все пробелы на &nbsp; """
    if not value:
        return ""
    return mark_safe(re.sub(r"\s", "&nbsp;", value))


PARTS_RE = re.compile(r"\s+-\s+")
PARTS_WRAP = '<span class="g-nowrap">%s</span>'


@jinja.filter
def mdash(value):
    """ Заменяет дефис на тире"""
    if not value:
        return ""

    parts = PARTS_RE.split(value)

    return u" — ".join(PARTS_WRAP % part for part in parts)


@jinja.filter
def mdash_wrappable(value):
    """ Заменяет дефис на тире, обертывает nowrap,
     оставляя возможность переноса в допустимых местах"""
    if not value:
        return ""

    parts = PARTS_RE.split(value)
    allowed_splitter = re.compile(ur'(?<=[^ ^][^ ^][^ ^]) (?!.{,3}[ \)\,\.$])', re.U | re.I | re.S)

    return u" — ".join([
        u' '.join([PARTS_WRAP % subpart for subpart in allowed_splitter.split(part)])
        for part in parts
    ])


@jinja.filter
def arrow(value):
    """ Заменяет дефис на стрелочку"""
    if not value:
        return ""
    return mark_safe(re.sub(r"\s+\-\s+", '<img src="%si/mobile/arrow.gif" alt="-" />' % settings.MEDIA_URL, value))


@jinja.filter
def nbsp_digits(str):
    """ неразрывный пробел после цифр """
    return mark_safe(re.sub(r'(?<=\d)\s', '&nbsp;', str))


@jinja.filter
def kill_br(str):
    """ убирает лишние <br>"""
    return str and mark_safe(re.sub('<br[^>]*?>', '', str)) or ""


@jinja.filter
def wrap(str, max_length):
    """ Урезает строку до max_length символов, добавляя многоточие """
    if not str:
        return ""
    if len(str) > max_length:
        return mark_safe(str[:max_length-3] + "&hellip;")
    else:
        return str


@jinja.filter
def cut_duration(str, make_cut):
    """ "1д 2 ч 51 мин" вместо "1 день 2 часа 21 минута". """
    if not make_cut:
        return str
    return re.sub(ur"д[^\s]+", u"д", re.sub(ur"ч[^\s]+", u"ч", re.sub(ur"м[^\s]+", u"мин", str)))


@jinja.filter
def shorten(str, do_shorten):
    if not do_shorten:
        return str
    return {u"отправление": u"отпр",
            u"прибытие": u"приб"
            }.get(str, str)


@jinja.filter
def truncate(str, length):
    if not str:
        return ""
    if len(str) <= length:
        return str
    return str[:length] + u"…"


def abbr(title):
    """ Сокращенное наименование города
    Первая буква слова и две следующие согласные
    """
    if not title:
        return ''
    if len(title) < 3:
        return title

    whitelist = u'цкнгшщзхждлрпвфчсмтб0123456789'

    abbr = title[0] + "".join(c for c in title[1:] if c in whitelist)[:2]

    if len(abbr) < 3:
        return title[:3]

    return abbr


@jinja.callable
def today():
    return date.today()


@jinja.callable
def class_attribute(*classes):
    """
    Вставляет аттрибут class(с пробелом перед ним)
    с переданными именами классов. Если переданы значения,
    приводящиеся к False, то они не добавляются.
    Если никаких правильных классов не передано, то возвращается
    пустая строка.
    """
    class_name = " ".join(cls for cls in classes if cls)

    if class_name:
        return ' class="%s"' % class_name

    return ''


@jinja.callable
def url(view_name, *args, **kwargs):
    urlconf = kwargs.pop('urlconf', None)

    from django.core.urlresolvers import reverse, NoReverseMatch

    try:
        return reverse(view_name, urlconf=urlconf, args=args, kwargs=kwargs)
    except NoReverseMatch:
        # Пробуем урлы основной версии
        return reverse(view_name, urlconf='urls', args=args, kwargs=kwargs)


class Positional(object):
    def __init__(self,  index, value):
        self.index = index
        self.value = value

    @classmethod
    def get(cls, request_get, name, index):
        try:
            return request_get.getlist(name)[index]
        except IndexError:
            return None


ONE_SHOT_PARAMS = ['noDirect']


@jinja.callable
def get_params(query_dict=None, escape=True, **params):
    """
    Параметры со значением None удаляются из исходных (если они там есть)
    """
    if query_dict:
        new_params = query_dict.copy()
        for param in ONE_SHOT_PARAMS:
            if param in new_params:
                del new_params[param]
    else:
        new_params = QueryDict(None, mutable=True)

    for name, value in params.items():
        if value is None:
            if name in new_params:
                del(new_params[name])
        elif isinstance(value, Positional):
            values = new_params.getlist(name)

            # Удлинняем массив, если в него не входит нужный индекс
            if len(values) < value.index + 1:
                values.extend([''] * (value.index + 1 - len(values)))

            values[value.index] = value.value
            new_params.setlist(name, values)
        elif isinstance(value, list):
            new_params.setlist(name, value)
        elif value is not False:
            new_params[name] = value

    query_string = new_params.urlencode()

    if escape:
        query_string = query_string.replace('&', '&amp;')

    return '?' + query_string


@jinja.filter
def set_to_json(s):
    return json.dumps(list(s), ensure_ascii=False).replace('"', "'")


class Sorter(object):
    """Сортировщик таблиц"""

    def __init__(self, columns, request_get, table_index=0, default=None):
        self._columns = columns
        self.request_get = request_get
        self.table_index = table_index

        # Сортировочная группа
        # FIXME: если на странице будет больше одного типа сортируемых
        # таблиц, нужно будет его как-то распределять
        self.group_index = 0

        # Попробуем вытащить из запроса
        try:
            requested_sort = request_get.getlist('sortBy')[self.group_index]
            if request_get.get('all_days_search') and 'tariff' in requested_sort:
                requested_sort = None
        except IndexError:
            requested_sort = None

        self.column = None
        self.reverse = False

        def parse(sort_by):
            return sort_by.startswith('-'), sort_by.lstrip('+-')

        for sort_by in [requested_sort, default]:
            if sort_by:
                reverse, column = parse(sort_by)

                try:
                    _, py_extractor = columns[column]

                    self.column = column
                    self.reverse = reverse

                    if callable(py_extractor):
                        self.extractor = py_extractor
                    else:
                        self.extractor = operator.itemgetter(py_extractor)

                    break

                except KeyError:
                    # нет такой колонки, ищем дальше
                    continue

        # Не нашли колонку, значит дефолтовая неправильная
        if not self.column:
            raise Exception("Invalid default sort specified")

        self.columns = self.ColumnAccessor(self)

    class ColumnAccessor(dict):
        """Как словарь возвращает данные для колонок"""

        def __init__(self, sorter):
            self.sorter = sorter

        def __getitem__(self, column):

            # KeyError вылезет наружу, что нам и нужно
            js_extractor, _ = self.sorter._columns[column]

            link = self.sorter.get_sort_params(column)
            direction = self.sorter.get_direction(column)

            return {
                'direction': direction,
                'link': link,
                'table_index': self.sorter.table_index,
                'metadata': json({
                    'tableIndex': self.sorter.table_index,
                    'column': column,
                    'extractor': js_extractor,
                    }),
                }

    def get_direction(self, column):
        if column == self.column:
            if self.reverse:
                return 'desc'
            else:
                return 'asc'

    def get_sort_params(self, column):
        # Если текущая сортировка является такой-же и прямой, то меняем направление
        if not self.reverse and self.column == column:
            new_sort = '-' + column
        else:
            # Иначе просто задаем нашу сортировку
            new_sort = '+' + column

        # Список сортировок для всех таблиц
        sort_list = self.request_get.getlist('sortBy')[:]

        # Удлинняем массив, если в него не входит нужная сортировка
        if len(sort_list) < self.group_index + 1:
            sort_list.extend([''] * (self.group_index + 1 - len(sort_list)))

        sort_list[self.group_index] = new_sort

        # Копируем параметры, чтобы внести изменения
        new_params = self.request_get.copy()

        new_params.setlist('sortBy', sort_list)

        return '?' + new_params.urlencode().replace('&', '&amp;')

    def sort(self, l):
        l.sort(key=self.extractor, reverse=self.reverse)

        # Значения с ключом None всегда должны быть в конце списка (RASP-1984)
        # поэтому если нет reverse, то нужно их туда переместить

        if not self.reverse:
            # Найдем, куда простирается граница None-значение (после сортировки
            # они оказываются все в начале списка)
            index = None
            for i, item in enumerate(l):
                if self.extractor(item) is not None:
                    # Первое не-None значение
                    index = i
                    break

            # Если не нашли не-None значение, то ничего менять не надо
            if index:
                # рокировка
                # [:] означает, что мы сам объект меняем
                l[:] = l[index:] + l[:index]

        # Поднимем интервальные нитки вверх
        def split_by_cond(arr, cond):
            pos = []
            neg = []
            for j in arr:
                if cond(j):
                    pos.append(j)
                else:
                    neg.append(j)
            return pos, neg

        interval, simple = split_by_cond(l, lambda s: getattr(s, 'thread', False) and s.thread.is_interval)
        interval.sort(key=lambda s: getattr(s, 'thread', False) and s.thread.begin_time or datetime(2001, 1, 1).time())
        l[:] = interval + simple


def json_encode(obj):
    if isinstance(obj, set):
        return list(obj)

    raise TypeError()


@jinja.filter  # noqa: F811
def json(data):
    return json.dumps(data, default=json_encode, ensure_ascii=False).replace('"', "'")


@jinja.filter
def cut_title(title):
    if not title:
        return ""
    return re.sub(u"область", u"обл.", title)


@jinja.callable
def schedule_change_alert():
    """ Предупреждение о смене расписаний [RASP-2554] """
    if date(2009, 5, 28) <= date.today() <= date(2009, 6, 10):
        return u'<span class="alert"><i class="b-icon"><i></i></i>Внимание, с 31 мая действует новый график движения!</span>'
    else:
        return ''


@jinja.callable
def search_url(city_from, city_to, date_forward=None, method='search', **extra):
    params = {
        'fromName': city_from.L_title(),
        'fromId': city_from.point_key,
        'toName': city_to.L_title(),
        'toId': city_to.point_key,
        }

    if date_forward:
        if isinstance(date_forward, date) or isinstance(date_forward, datetime):
            params['when'] = human_date_without_year(date_forward)
        else:
            params['when'] = date_forward

    params.update(extra)

    # Убираем None
    params = dict((k, v) for k, v in params.items() if v is not None)

    return "%s?%s" % (url(method), urlencode(params).replace('&', '&amp;'))


@jinja.callable
def suburban_search_url(city_from, city_to, next=False, **extra):
    params = {
        'fromName': city_from.L_title(),
        'fromId': city_from.point_key,
        'toName': city_to.L_title(),
        'toId': city_to.point_key,
        }

    params.update(extra)

    path = '/search/suburban/'

    if next:
        path += 'next/'

    return "%s?%s" % (path, urlencode(params))


@jinja.callable
def time_correction():
    return '<script type="text/javascript" src="/time/correction?%s"></script>' % time.time()


@jinja.filter
def parse_phones(text):
    u""" Ищет в тексте телефоны и заворачивает в специальные ссылки [MOBRASP-256] """
    def convert(result):
        number = result.group(1)
        clean = re.sub(r'[^\d]', '', number)
        return '<a href="tel:%s">%s</a>' % (clean, number)
    text = re.sub(r'((\+|8)[\d\s()\.-]+\d)', convert, text)
    return text


@jinja.callable
def qadjectiveforms(count, form0, form1):
    count = int(count)

    if count % 100 == 11:
        return form1
    elif count % 10 == 1:
        return form0
    else:
        return form1


@jinja.callable
def qnounforms(count, form0, form1, form2):
    count = int(count)

    if count % 100 in [11, 12, 13, 14]:
        return form0
    elif count % 10 == 1:
        return form1
    elif count % 10 in [2, 3, 4]:
        return form2
    else:
        return form0


@jinja.callable
def media_url():
    return settings.MEDIA_URL


@jinja.filter
def js_date(dt):
    return "new Date('%s')" % dt.strftime('%B %d, %Y %H:%M:%S')


@jinja.filter
def cut_direction(title):
    return title.replace(u'направление', u'напр.')


@jinja.callable
def extend(a, *b):
    """Mimics jQuery extend"""

    for d in b:
        a.update(d)

    return a


@jinja.callable
def make_list(iter):
    return list(iter)


@jinja.callable
def advert_link_domain(domain):
    return {'kz': 'kz', 'com': 'com'}.get(domain, 'ru')


@jinja.callable
def current_domain(request):
    host = request.META['HTTP_HOST'].lower()
    if ':' in host:
        host, port = host.split(':')

    return host[host.index('yandex') + 7:]


@jinja.filter
def smart_escape(s):
    return mark_for_escaping(s)


@jinja.filter
def nice_quotes(value):
    """ Заменяет кавычки-лапки на кавычки-ёлочки"""
    if not value:
        return u''

    parts = value.split(u'"')

    # Меняем только если в строке две кавычки
    if len(parts) == 3:
        return parts[0] + u'«' + parts[1] + u'»' + parts[2]
    else:
        return value
