# -*- coding: utf-8 -*-
from __future__ import absolute_import

import json
from datetime import datetime, date

import pytz
from django.core.serializers.json import DjangoJSONEncoder

from travel.avia.admin.www.templatetags.common import class_attribute
from travel.avia.library.python.common.utils.date import get_pytz, FuzzyDateTime, human_date_without_year, human_date_with_time
from travel.avia.admin.lib.jinja import Library

jinja = Library()


@jinja.callable
def cities_get_param(tz_cities):
    return ','.join(str(city.id) for city in tz_cities)


def json_dt_struct(dt, tz_cities, col=None):
    shifts = {}

    local_dt = dt.replace(tzinfo=None)

    for city in tz_cities:
        delta = dt.astimezone(get_pytz(city.time_zone)).replace(tzinfo=None) - local_dt
        shifts[city.id] = delta.seconds / 60 + delta.days * 24 * 60

    data = {
        'local': dt.strftime("%B %d, %Y %H:%M:%S"),
        'shifts': shifts
    }

    if col:
        data['col'] = col

    if isinstance(dt, FuzzyDateTime):
        data['fuzzy'] = True

    return data


@jinja.filter
def json_dt_list(dts, tz_cities):
    return json.dumps([json_dt_struct(dt, tz_cities) for dt in dts], cls=DjangoJSONEncoder).replace('"', "'")


@jinja.callable
def json_dt(dt, tz_cities, col=None):
    """cities - объекты, во временных зонах которых нужно будет отображать дату"""
    # Сдвиги относительно локального времени для требуемых временных зон.
    # Чтобы получить требуемую временную зону, надо её сдвиг прибавить к локальной дате.
    if not dt:
        return '{}'

    return json.dumps(json_dt_struct(dt, tz_cities, col), cls=DjangoJSONEncoder).replace('"', "'")


@jinja.filter
def astimezone(dt, city):
    # tz передается как город
    if not city:
        # местное время
        return dt

    return dt.astimezone(get_pytz(city.time_zone))


@jinja.filter
def js_wrap(dt, format, tz, tz_cities, column=None, wrapper="span", classes=[]):
    """
    Дата-время, обернутые в тэг, с метаданными
    для переключения временной зоны
    """
    formatters = {
        'js-time-only': lambda dt: dt.strftime(u'%H:%M'),
        'single': human_date_with_time,
        'js-date-only': human_date_without_year,
        'js-blank': lambda unused: u'',
        }

    if format not in formatters:
        raise ValueError("Unknown datetime format %s" % format)

    cls = class_attribute('js-datetime', format,
                          *(classes + [json_dt(dt, tz_cities, column)]))

    formatted_dt = formatters[format](astimezone(dt, tz))

    return u"""<%s%s>%s</%s>""" % (wrapper, cls, formatted_dt, wrapper)


@jinja.filter
def js_wrap_time_ontop(tm, format, tz, *args, **kwargs):
    """
    Время, дополненное special-case датой, поднимаемой при сортировке в начало
    обернутое в тэг, с метаданными
    для переключения временной зоны
    """
    dt = datetime.combine(date(1981, 1, 12), tm).replace(tzinfo=pytz.UTC)
    return js_wrap(dt, format, tz, *args, **kwargs)


@jinja.callable
def js_timezone_abbr(local, current):
    local_tz_name = local.tz_name_abbr

    if current:
        current_tz_name = current.tz_name_abbr
    else:
        current_tz_name = local_tz_name

    return '<span class="js-time-zone {local: \'%s\'}">%s</span>' % (local_tz_name, current_tz_name)


@jinja.callable
def days_texts(thread, start_date, dt, time_zone, tz_cities, *args, **kwargs):
    shifts = {}

    if time_zone not in tz_cities:
        tz_cities.append(time_zone)

    for tz in [None] + tz_cities:
        shift = (astimezone(dt, tz).date() - start_date).days

        shifts.setdefault(shift, set()).add(tz)

    for shift, timezones in shifts.items():
        active = time_zone in timezones

        yield timezones, thread.L_days_text(shift, *args, **kwargs), active
