# coding: utf8

from __future__ import absolute_import

import logging
from django.db import connection, transaction

from travel.avia.library.python.common.models.schedule import Route, RThread
from travel.avia.admin.lib.iterators import chunker


log = logging.getLogger(__name__)


ID_CHUNK_SIZE = 10000
UID_CHUNK_SIZE = 3000


@transaction.atomic
def fast_delete_threads(qs, log=None):
    assert qs.model == RThread

    thread_ids = list(qs.values_list('id', flat=True))

    if not thread_ids:
        return

    cursor = connection.cursor()

    if log is not None:
        log.info(u'Удаляем нитки')

    while thread_ids:
        chunk_ids = thread_ids[:ID_CHUNK_SIZE]
        thread_ids = thread_ids[ID_CHUNK_SIZE:]

        delete_threads_from_z_tables(chunk_ids, cursor)

        del_sql = u' '.join([
            u'DELETE t, rts FROM www_rthread t LEFT JOIN www_rtstation rts ON rts.thread_id = t.id',
            u'WHERE t.id IN (',
            u','.join(['%s'] * len(chunk_ids)),
            u')'
        ])

        cursor.execute(del_sql, chunk_ids)

    if log is not None:
        log.info(u'Нитки удалены')


@transaction.atomic
def fast_delete_package_threads_and_routes(two_stage_package, log=None):
    routes = list(Route.objects.filter(two_stage_package_id=two_stage_package.id).values_list('id', 'script_protected'))
    affected_route_ids = [route_id for route_id, script_protected in routes if not script_protected]
    excluded_route_ids = [route_id for route_id, script_protected in routes if script_protected]

    affected_thread_ids = list(RThread.objects.filter(route_id__in=affected_route_ids).values_list('id', flat=True))

    if not affected_thread_ids:
        return

    if log is not None:
        log.info(u'Удаляем нитки')

    cursor = connection.cursor()

    # Z tables
    del_znoderoute_sql = 'DELETE FROM www_znoderoute2 WHERE two_stage_package_id = %s'
    if excluded_route_ids:
        del_znoderoute_sql += 'AND route_id NOT IN ({})'.format(','.join(['%s'] * len(excluded_route_ids)))
    cursor.execute(del_znoderoute_sql,  [two_stage_package.id] + excluded_route_ids)

    for thread_ids_chunk in chunker(affected_thread_ids, ID_CHUNK_SIZE):
        in_clause = "IN(%s)" % (u",".join(map(str, thread_ids_chunk)))
        cursor.execute("""DELETE FROM www_stationschedule WHERE thread_id %s""" % in_clause)

    # threads
    for thread_ids_chunk in chunker(affected_thread_ids, ID_CHUNK_SIZE):
        del_sql = u' '.join([
            u'DELETE t, rts FROM www_rthread t LEFT JOIN www_rtstation rts ON rts.thread_id = t.id',
            u'WHERE t.id IN (',
            u','.join(['%s'] * len(thread_ids_chunk)),
            u')'
        ])
        cursor.execute(del_sql, thread_ids_chunk)

    # routes
    del_route_sql = 'DELETE FROM www_route WHERE script_protected = 0 AND two_stage_package_id = %s'
    cursor.execute(del_route_sql, [two_stage_package.id])


@transaction.atomic
def fast_delete_supplier_threads_and_routes(supplier, log=None):
    routes = list(Route.objects.filter(supplier_id=supplier.id, two_stage_package_id=None)
                               .values_list('id', 'script_protected'))
    affected_route_ids = [route_id for route_id, script_protected in routes if not script_protected]
    excluded_route_ids = [route_id for route_id, script_protected in routes if script_protected]

    affected_thread_ids = list(RThread.objects.filter(route_id__in=affected_route_ids).values_list('id', flat=True))

    if not affected_thread_ids:
        return

    if log is not None:
        log.info(u'Удаляем нитки')

    cursor = connection.cursor()

    # Z tables
    del_znoderoute_sql = 'DELETE FROM www_znoderoute2 WHERE supplier_id = %s AND two_stage_package_id is NULL'
    if excluded_route_ids:
        del_znoderoute_sql += 'AND route_id NOT IN ({})'.format(','.join(['%s'] * len(excluded_route_ids)))
    cursor.execute(del_znoderoute_sql,  [supplier.id] + excluded_route_ids)

    for thread_ids_chunk in chunker(affected_thread_ids, ID_CHUNK_SIZE):
        in_clause = "IN(%s)" % (u",".join(map(str, thread_ids_chunk)))
        cursor.execute("""DELETE FROM www_stationschedule WHERE thread_id %s""" % in_clause)

    # threads
    for thread_ids_chunk in chunker(affected_thread_ids, ID_CHUNK_SIZE):
        del_sql = u' '.join([
            u'DELETE t, rts FROM www_rthread t LEFT JOIN www_rtstation rts ON rts.thread_id = t.id',
            u'WHERE t.id IN (',
            u','.join(['%s'] * len(thread_ids_chunk)),
            u')'
        ])
        cursor.execute(del_sql, thread_ids_chunk)

    # routes
    del_route_sql = 'DELETE FROM www_route WHERE script_protected = 0 AND supplier_id = %s AND two_stage_package_id is NULL'
    cursor.execute(del_route_sql,  [supplier.id])

    if log is not None:
        log.info(u'Нитки удалены')


@transaction.atomic
def fast_delete_routes(qs, log=None):
    assert qs.model == Route
    route_ids = list(set(qs.values_list('id', flat=True)))

    if not route_ids:
        return

    if log is not None:
        log.info(u'Удаляем маршруты')

    cursor = connection.cursor()

    while route_ids:
        chunk_ids = route_ids[:ID_CHUNK_SIZE]
        route_ids = route_ids[ID_CHUNK_SIZE:]

        delete_routes_from_z_tables(chunk_ids, cursor)

        del_sql = u' '.join([
            u'DELETE r, t, rts FROM www_route r LEFT JOIN www_rthread t ON t.route_id = r.id',
            u'LEFT JOIN www_rtstation rts ON rts.thread_id = t.id',
            u'WHERE r.id IN (',
            u','.join(['%s'] * len(chunk_ids)),
            u')'
        ])

        cursor.execute(del_sql, chunk_ids)

    if log is not None:
        log.info(u'Маршруты удалены')


@transaction.atomic
def delete_routes_from_z_tables(route_ids, cursor):
    in_clause = "IN(%s)" % (u",".join(map(str, route_ids)))

    cursor.execute("""DELETE FROM www_znoderoute2 WHERE route_id %s""" % in_clause)

    cursor.execute("""DELETE FROM www_stationschedule WHERE route_id %s""" % in_clause)


@transaction.atomic
def delete_threads_from_z_tables(thread_ids, cursor):
    in_clause = "IN(%s)" % (u",".join(map(str, thread_ids)))

    cursor.execute("""DELETE FROM www_znoderoute2 WHERE thread_id %s""" % in_clause)

    cursor.execute("""DELETE FROM www_stationschedule WHERE thread_id %s""" % in_clause)


@transaction.atomic
def correct_z_tables(cursor):
    cursor.execute("""DELETE z FROM www_znoderoute2 z LEFT JOIN www_rthread t ON t.id = z.thread_id
                        WHERE t.id IS NULL""")
    cursor.execute("""DELETE z FROM www_znoderoute2 z LEFT JOIN www_route r ON r.id = z.route_id
                    WHERE r.hidden = 1 OR r.id IS NULL""")

    cursor.execute("""DELETE ss FROM www_stationschedule ss LEFT JOIN www_route r ON r.id = ss.route_id
                    WHERE r.hidden=1 OR r.id IS NULL
                    """)
    cursor.execute("""DELETE ss FROM www_stationschedule ss LEFT JOIN www_rthread t ON t.id = ss.thread_id
                    WHERE t.id IS NULL""")

    cursor.execute("""DELETE rni FROM www_routenumberindex rni
                    LEFT JOIN www_route r ON r.id = rni.route_id
                    WHERE r.id IS NULL""")


@transaction.atomic
def fast_delete_tariffs_by_uids(thread_uids):
    thread_uids = list(thread_uids)

    cursor = connection.cursor()

    while thread_uids:
        chunk_uids = thread_uids[:UID_CHUNK_SIZE]
        thread_uids = thread_uids[UID_CHUNK_SIZE:]

        del_sql = u' '.join([
            u'DELETE FROM www_threadtariff',
            u'WHERE thread_uid IN (',
            u','.join(['%s'] * len(chunk_uids)),
            u')'
        ])

        cursor.execute(del_sql, chunk_uids)
