# -*- coding: utf-8 -*-
from __future__ import absolute_import

import re
import hashlib
from urllib import quote

from travel.avia.admin.lib.exceptions import SimpleUnicodeException
from travel.avia.library.python.common.models.schedule import RThreadType
from travel.avia.admin.www.utils.data import rus2translit


THREAD_UUID_FORMAT = u'{route_uid}-{thread_type}-{thread_params_md5}-{stops_md5}'

CLEAN_UID_RE = re.compile(ur'[^-a-zA-Z_0-9*]')
REPLACEMENT = '*'

BAD_THREAD_TYPE_IDS_FOR_ROUTE_UID = {RThreadType.CHANGE_ID, RThreadType.CANCEL_ID}


class RouteUidGenerationError(SimpleUnicodeException):
    pass


def gen_route_uid(thread, use_start_station=False, use_company=False, use_stations=False, red_metaroute_id=None,
                  use_schedule_plan=True, thread_type_check=True):
    """
    Генерация route_uid для рейса учитывая поставщика и
    начальную станцию, или начальнцю и конечную станции, если это необходимо
    Первая часть route_uid это номер рейса
    """

    # TODO удалить assert после переноса метаданных из рейса в нитку
    # assert сейчас нужен, чтобы убедится, что теперь все передают нитку для генерации уида рейса
    from travel.avia.library.python.common.models.schedule import RThread
    assert isinstance(thread, RThread)

    if thread_type_check and thread.type_id in BAD_THREAD_TYPE_IDS_FOR_ROUTE_UID:
        raise RouteUidGenerationError(u'Неподходящий тип нитки %s для построения route_uid', thread.type.title)

    uid_parts = []

    uid_parts.append(transform_number_to_uid_part(thread))

    if use_start_station or use_stations:
        try:
            rtstations = thread.rtstations
        except AttributeError:
            rtstations = list(thread.rtstation_set.all())

        if not rtstations:
            raise RouteUidGenerationError(u'Невозможно построить route_uid без начальной станции у нитки')

        if use_start_station:
            uid_parts.append(unicode(rtstations[0].station_id))

        if use_stations:
            uid_parts.append(u'f%dt%d' % (rtstations[0].station_id, rtstations[-1].station_id))

    if use_company and thread.company_id:
        uid_parts.append(u'c' + unicode(thread.company_id))

    if use_schedule_plan and thread.schedule_plan_id:
        uid_parts.append(thread.schedule_plan.code)

    if red_metaroute_id:
        uid_parts.append(u'r%d' % red_metaroute_id)

    uid_parts.append(unicode(thread.supplier_id))

    route_uid = u'_'.join(uid_parts)

    # clean
    route_uid = route_uid.replace(u'/', u'x').replace(u' ', u'-')

    return CLEAN_UID_RE.sub(REPLACEMENT, route_uid)


def transform_number_to_uid_part(thread):
    number = thread.number or thread.hidden_number

    if not number:
        return u'empty'

    number = rus2translit(number)

    return quote(number.encode('utf8'), safe='/ ').replace('%', REPLACEMENT).decode('utf8')


def gen_thread_uid(thread):
    """ Генерация thread_uid на основе номера нитки и route_uid """

    uid_parts = thread.route.route_uid.split(u'_')

    # Первая часть route_uid это номер рейса
    uid_parts.insert(1, unicode(thread.ordinal_number))

    if thread.supplier_id and thread.supplier_id != thread.route.supplier_id:
        uid_parts.append(u"_as-%s" % thread.supplier_id)

    return u'_'.join(uid_parts)


def gen_thread_uuid(thread):
    valuable_persistent_attrs = (
        'title',
        'is_manual_title',
        'title_tr',
        'title_tr_author',
        'title_common',
        'title_short',
        't_model_id',
        'is_circular',
        't_subtype_id',
        'supplier_id',
        'comment',
        'density',
        'tz_start_time',
        'time_zone',
        't_type_id',
        'company_id',
        'number',
        'hidden_number',
        'ordinal_number',
        'express_type',
        'express_lite_id',
        'schedule_plan_id',
        'template_text',
        'template_start',
        'template_end',
        'template_code',
        #  Для интервальных ниток
        'period_int',
        'begin_time',
        'end_time',
        'density',
        'comment',
    )

    params = dict()
    params['route_uid'] = thread.route.route_uid
    params['thread_type'] = thread.type_id

    thread_params = []
    for attr in valuable_persistent_attrs:
        value = getattr(thread, attr, u"")
        value = unicode(value) if value else u""

        thread_params.append(value)

    params['thread_params_md5'] = hashlib.md5(u'<>'.join(thread_params).encode('utf8'))\
        .hexdigest()

    rtstations = thread.get_rtstations()

    # Приводим типы, чтобы генерация из базы совпадала с генерацией во время импорта
    stations = [
        u'-'.join([
            unicode(rts.station_id),
            unicode(rts.terminal_id) if rts.terminal_id else u'None',
            unicode(rts.platform),

            unicode(int(rts.tz_arrival)) if rts.tz_arrival is not None else u'None',
            unicode(int(rts.tz_departure)) if rts.tz_departure is not None else u'None',
            unicode(rts.time_zone),

            unicode(bool(rts.arrival_code_sharing)),
            unicode(bool(rts.departure_code_sharing)),

            unicode(bool(rts.is_technical_stop)),

            # Fuzzy flags
            unicode(bool(rts.is_fuzzy)),
            unicode(bool(rts.is_searchable_to)),
            unicode(bool(rts.is_searchable_from)),
            unicode(bool(rts.in_station_schedule)),
            unicode(bool(rts.in_thread)),
        ]) for rts in rtstations
    ]

    stations = u':'.join(stations)

    params['stops_md5'] = hashlib.md5(stations.encode('utf8')).hexdigest()

    return THREAD_UUID_FORMAT.format(**params)
