# -*- coding: utf-8 -*-

import os
import logging
import re
from datetime import datetime, timedelta

from django.conf import settings
from django.db import connection, OperationalError
from django.http import HttpResponse, HttpResponseServerError


logger = logging.getLogger(__name__)


def _check_db():
    try:
        connection.ensure_connection()
    except OperationalError:
        logger.exception('Database ERROR')
        return False

    return True


def version(request):
    return HttpResponse(settings.PKG_VERSION, content_type="text/plain")


def exception(request):
    return 1 / 0


def django_ping(request):
    if _check_db():
        return HttpResponse("Django is alive", content_type="text/plain")
    else:
        return HttpResponseServerError("MYSQL Error", content_type="text/plain")


def read_backwards(path):
    BUFSIZE = 1024

    f = open(path, 'r')

    f.seek(0, os.SEEK_END)

    pos = f.tell()

    line = ''

    while True:
        if pos > BUFSIZE:
            to_read = BUFSIZE
        else:
            to_read = pos

        pos -= to_read

        f.seek(pos)

        buf = f.read(to_read)

        buf_lines = buf.split('\n')

        buf_lines[-1] = buf_lines[-1] + line

        for l in reversed(buf_lines[1:]):
            yield l

        line = buf_lines[0]

        if pos == 0:
            yield line

            break

    f.close()


RE = re.compile(r'\d{4}-\d\d-\d\d \d\d:\d\d')
SANE_LIMIT = 10000


def logtail(request, path):
    log_path = os.path.join(settings.LOG_PATH, path)

    try:
        interval = timedelta(seconds=int(request.GET['interval']))
    except (KeyError, ValueError):
        return HttpResponseServerError("Required parameter missing or invalid", content_type="text/plain")

    lines = []

    for line in read_backwards(log_path):
        m = RE.search(line)

        if m:
            dt = datetime.strptime(m.group(0), "%Y-%m-%d %H:%M")

            if request.now - dt > interval:
                break

        lines.append(line)

        if len(lines) > SANE_LIMIT:
            break

    return HttpResponse("\n".join(reversed(lines)), content_type="text/plain")
