# -*- coding: utf-8 -*-

from django.conf.urls import url
from django.db import models
from django.utils.translation import ugettext_lazy as _

from travel.avia.library.python.common.models_abstract.geo import BaseStation, BaseDistrict

from travel.avia.library.python.common.utils.locations import get_params
from travel.avia.library.python.common.utils.text import cut_direction


class Station(BaseStation):
    """ Станция """

    is_base = models.BooleanField(_(u'Является базовым автовокзалом'), default=False,
                                  help_text=_(u'Признак базового автовокзала (таск RASPADMIN-183)'))

    def omonim_title(self, request, direction, extra_params=None, for_html=False):
        """ Название станции со ссылкой для переключения в омонимах
            direction - (From|To)
            current - является выбранным элементом
        """
        result = {
            'title': self.title_with_prefix,
            'add': '',
        }

        add = []
        if extra_params is None:
            extra_params = {}

        disputed = self.is_disputed()

        if self.district:
            add.append(u"%s р-н" % self.district.title)

        if self.settlement_id:
            if (hasattr(self, 'hide_settlement') or extra_params.get('hide_settlement')) or not for_html:
                add.append(self.settlement.title)
            else:
                params = get_params(request.GET, **{direction + 'Id': self.settlement.point_key,
                                                    direction + 'Name': self.settlement.title})
                add.append(u"<a class='b-link' href=\"%s\">%s</a>" % (params, self.settlement.title_with_prefix))

        if self.region_id and (self.t_type_id == 2 or not self.settlement_id):
            add.append(self.region.title)

        if self.country_id and (self.t_type_id == 2 or not (self.settlement_id or self.region_id)):
            if not disputed:
                add.append(self.country.title)

        if hasattr(self, 'force_show_region') and self.region_id:
            add.append(self.region.title)

        if hasattr(self, 'show_direction'):
            try:
                direction = self.externaldirectionmarker_set.all().select_related('external_direction')[
                    0].external_direction
            except IndexError:
                pass
            else:
                s_id = self.settlement_id or (direction.suburban_zone and direction.suburban_zone.settlement_id) or None
                if s_id and for_html:
                    add.append(
                        u'<a class="b-link" href="%s" title="%s">%s</a>' % (
                            url('city_direction', s_id) +
                            '?direction=' + direction.code,
                            cut_direction(direction.full_title),
                            cut_direction(direction.full_title)
                        )
                    )
                else:
                    # Не могу определить город направления
                    add.append(cut_direction(direction.full_title))

        if add:
            result['add'] = u", ".join(add)

        return result

    @property
    def is_mta(self):
        # МТА станции определяем по соответствию импорта или по рейсам
        return (
            self.stationmapping_set.filter(supplier__code='mta').count() > 0
            or self.rtstation_set.filter(thread__route__supplier__code='mta').count() > 0
        )

    @classmethod
    def autotranslate(cls, stations):
        # FIXME: Кажется это хрень уже не работает
        from travel.avia.library.python.common.utils.translate import translate, TranslateError
        from travel.avia.admin.precalc.utils import TheEstimator

        estimator = TheEstimator(stations.count())

        for station in stations:
            for field, languages in cls.AUTOTRANSLATE.items():
                for lang in languages:
                    attr = '%s_%s' % (field, lang)

                    if getattr(station, attr):
                        # Перевод уже есть
                        continue

                    try:
                        translated = translate(lang, station.title)
                    except TranslateError:
                        continue

                    setattr(station, attr, translated)
                    setattr(station, attr + '_author', 'translate')

            station.save()

            estimator.update(1)

    class Meta(BaseStation.Meta):
        app_label = 'www'


class District(BaseDistrict):
    class Meta:
        app_label = 'www'

    def get_log(self):
        from travel.avia.admin.importinfo.models import RelatedLog

        related_log = RelatedLog.get_by_object(self)

        if related_log:
            return related_log.log
