# -*- coding: utf-8 -*-

import json
import os
import os.path
import re
from textwrap import dedent

from lxml import etree

from django.conf import settings
from django.utils import translation

from babel.messages.plurals import get_plural

from travel.avia.library.python.common.utils.safe_xml_parser import safe_xml_fromstring

KEYSET_PATH = os.path.join(os.path.dirname(__file__), 'keyset.json')

_keyset = None
_keyset_stat = None


def _keyset_changed():
    try:
        stat = os.stat(KEYSET_PATH)
    except:
        return True

    for attr in ['st_size', 'st_mtime']:
        if getattr(stat, attr) != getattr(_keyset_stat, attr):
            return True

        return False


def get_keyset():
    global _keyset, _keyset_stat

    if not _keyset or _keyset_changed():
        try:
            with open(KEYSET_PATH) as f:
                _keyset_stat = os.stat(KEYSET_PATH)
                _keyset = json.load(f)
        except:
            _keyset = None

    return _keyset or {}


KEY_RE = re.compile(r"[\n ]+", re.U)


def form_key(form):
    return KEY_RE.sub(' ', form).strip()


def _gettext(form, key=None, lang=None):
    lang = lang or translation.get_language()

    form = dedent(form).strip()

    if not key:
        key = form_key(form)

    try:
        data = get_keyset()[key]
    except KeyError:
        return form

    assert not data["info"]["is_plural"]

    try:
        key_translation = data["translations"][lang]
    except KeyError:
        return form

    translated_form = key_translation["form"]

    if translated_form:
        return translated_form

    return form


def _ngettext(n, *forms, **kwargs):
    lang = translation.get_language()

    forms = [dedent(form).strip() for form in forms]

    num_plurals, func = get_plural_func(settings.BASE_LANGUAGE)

    assert len(forms) == num_plurals

    base_form = forms[func(n)]

    key = kwargs.pop('key', None)

    if kwargs:
        raise ValueError("Only allowed keyword argument is key")

    if not key:
        key = form_key(forms[0])

    try:
        data = get_keyset()[key]
    except KeyError:
        return base_form

    assert data["info"]["is_plural"]

    try:
        key_translation = data["translations"][lang]
    except KeyError:
        return base_form

    num_plurals, func = get_plural_func(lang)

    form_n = func(n)

    translated_form = key_translation["form%d" % (form_n + 1)]

    if translated_form:
        return translated_form

    return base_form


def marktrans(form):
    if getattr(translation, 'marktrans', False):
        if isinstance(form, basestring):
            return '[trans]%s[/trans]' % form

        return ['[trans]', form, '[/trans]']

    return form


def markdbtrans(form):
    if getattr(translation, 'marktrans', False):
        if isinstance(form, basestring):
            return '[dbtrans]%s[/dbtrans]' % form

        return ['[dbtrans]', form, '[/dbtrans]']

    return form


def gettext(form, **kwargs):
    return marktrans(_gettext(form, **kwargs))


class LazyForm(object):
    def __init__(self, form):
        self.form = form


def mark_gettext(form, key=None):
    """
    Помечает константу для выгрузки в tanker

    Локализованное значение таких констант должно получаться методом dynamic_gettext
    """
    return form


def dynamic_gettext(form, **kwargs):
    """
    Производит локализацию константы ранее помеченной методом mark_gettext.
    """
    return marktrans(_gettext(form, **kwargs))


_plural_func_cache = {}


def get_plural_func(lang):
    import hemi

    try:
        return _plural_func_cache[lang]
    except KeyError:
        pass

    plural = get_plural(lang)

    source = "(function(n) { return %s })" % plural.plural_expr

    func = hemi.Context().eval(source)

    _plural_func_cache[lang] = plural.num_plurals, func

    return _plural_func_cache[lang]


def ngettext(n, *forms, **kwargs):
    return marktrans(_ngettext(n, *forms, **kwargs))


DOCTYPE_STRING = """<!DOCTYPE html [
    <!ENTITY nbsp   "&#160;"> <!-- no-break space = non-breaking space, U+00A0 ISOnum -->
    <!ENTITY laquo  "&#171;"> <!-- left-pointing double angle quotation mark = left pointing guillemet, U+00AB ISOnum -->
    <!ENTITY raquo  "&#187;"> <!-- right-pointing double angle quotation mark = right pointing guillemet, U+00BB ISOnum -->
    <!ENTITY mdash   "&#8212;"> <!-- em dash, U+2014 ISOpub -->
]>"""
XML_HEADER = '<?xml version="1.0" standalone="yes"?>'


def xgettexttree(form):
    xml = XML_HEADER + DOCTYPE_STRING + '<xml>' + form + '</xml>'
    return safe_xml_fromstring(xml, resolve_entities=True)


def xformat(form, is_text=False, **kwargs):
    root = xgettexttree(form)

    def process(element):
        content = []

        if element.text:
            content.append(element.text)

        for child in element:
            child_content = process(child)

            handler_name = child.tag.replace('-', '_')

            if handler_name in kwargs:
                handler = kwargs[handler_name]

                if child_content:
                    child_content = handler(child_content, **child.attrib)

                elif child.attrib or callable(handler):
                    child_content = handler(**child.attrib)

                else:
                    child_content = handler

            else:
                elem = etree.Element(child.tag, child.attrib)

                elem.text = ''

                elemxml = etree.tostring(elem)

                split_index = elemxml.index('><') + 1

                child_content = [elemxml[:split_index], child_content, elemxml[split_index:]]

            content.append(child_content)

            if child.tail:
                content.append(child.tail)

        return u''.join(content) if is_text else content

    return process(root)


def xgettext(form, **kwargs):
    """
    Возвращает BEM-json содержащий перевод для form

    Пример:
    tgettext("Это '<word />' слово", **{ 'word': 'слово' })

    Результат:
    ["Это '", "слово", "' слово"]
    """

    key = kwargs.pop('key', None)
    lang = kwargs.pop('lang', None)

    form = _gettext(form, key=key, lang=lang)

    return marktrans(xformat(form, **kwargs))


def tgettext(form, **kwargs):
    """
    Возвращает строку содержащую перевод для form

    Пример:
    tgettext("Это '<word />' слово", **{ 'word': 'слово' })

    Результат:
    Это 'слово' слово
    """

    key = kwargs.pop('key', None)
    lang = kwargs.pop('lang', None)

    form = _gettext(form, key=key, lang=lang)

    return marktrans(xformat(form, is_text=True, **kwargs))


def xngettext(n, *forms, **kwargs):
    """
    Возвращает BEM-json содержащий перевод для forms с учетом plural-форрмы

    Пример:
    xngettext(1, "<n /> слово", "<n /> слова", "<n /> слов")

    Результат:
    ["1", " слово"]

    Пример:
    xngettext(2, "<n /> слово", "<n /> слова", "<n /> слов")

    Результат:
    ["2", " слова"]

    Пример:
    xngettext(25, "<n /> слово", "<n /> слова", "<n /> слов")

    Результат:
    ["25", " слов"]
    """

    form = _ngettext(n, *forms, key=kwargs.pop('key', None))

    kwargs["n"] = n

    return marktrans(xformat(form, **kwargs))


def tngettext(n, *forms, **kwargs):
    """
    Возвращает строку содержащую перевод для forms с учетом plural-форрмы

    Пример:
    xngettext(1, "<n /> слово", "<n /> слова", "<n /> слов")

    Результат:
    "1 слово"

    Пример:
    xngettext(2, "<n /> слово", "<n /> слова", "<n /> слов")

    Результат:
    "2 слова"

    Пример:
    xngettext(25, "<n /> слово", "<n /> слова", "<n /> слов")

    Результат:
    "25 слов"
    """

    form = _ngettext(n, *forms, key=kwargs.pop('key', None))

    kwargs["n"] = n

    return marktrans(xformat(form, is_text=True, **kwargs))
