import logging
from datetime import datetime
from asyncio import run

import ujson
from lxml import etree

from travel.avia.aeroflot_queue_proxy import settings
from travel.avia.aeroflot_queue_proxy.lib.remove_personal_data import remove_personal_data
from travel.avia.aeroflot_queue_proxy.lib.yt_loggers.orders.aeroflot_queue_order_message import (
    log_aeroflot_queue_order_message,
)
from travel.avia.aeroflot_queue_proxy.service.ibm_mq import IBMMQClient
from travel.avia.aeroflot_queue_proxy.service.sqs import sqs_client

logger = logging.getLogger(__name__)


def _send_message(date_sent, marker, message, queue_url, xml, queue_is_fifo=True):
    try:
        send_message_kwargs = dict(
            QueueUrl=queue_url,
            MessageBody=ujson.dumps({
                'parsed': xml is not None,
                'marker': marker,
                'date_sent': date_sent.strftime('%Y-%m-%d %H:%M:%S'),
                'date_received': datetime.now().strftime('%Y-%m-%d %H:%M:%S'),
                'message': etree.tostring(xml) if xml is not None else message,
            }),
        )
        if queue_is_fifo:
            send_message_kwargs.update(
                MessageDeduplicationId='{}_{}'.format(marker, date_sent.strftime('%Y%m%d%H%M%S')),
                MessageGroupId=marker,
            )
        message_id = sqs_client.send_message(**send_message_kwargs)
    except Exception:
        logger.exception(
            'Error sending message for marker "%s", queue: %s',
            marker, queue_url,
        )
        raise
    else:
        logger.info(
            'Message for marker "%s" sent, id: %s, queue: %s',
            marker, message_id, queue_url,
        )


async def _handle_message(message, queue_url_clickout: str, queue_url_cpa: str, queue_url_boy: str):
    xml = None
    stripped_message_as_string = remove_personal_data(message.decode(encoding='utf-8'))
    stripped_message_as_bytes = stripped_message_as_string.encode('utf-8')
    logger.info('Message: %s', stripped_message_as_bytes)
    try:
        xml = etree.fromstring(stripped_message_as_bytes)
        date_sent = datetime.strptime(xml.find('header').findtext('dateSent'), '%Y-%m-%d %H:%M:%S')
        marker = xml.xpath('//content/bookings/booking/ServiceLines')[0].findtext('REFERRER')
        await log_aeroflot_queue_order_message(stripped_message_as_string, marker)
    except Exception as e:
        logger.exception('Error parsing xml (%s): %s', e, stripped_message_as_bytes)
        date_sent = datetime.now()
        marker = 'None'

    if marker in settings.AEROFLOT_BOY_MARKERS_LIST:
        _send_message(date_sent, marker, stripped_message_as_bytes, queue_url_boy, xml)
    else:
        _send_message(date_sent, marker, stripped_message_as_bytes, queue_url_clickout, xml)
        _send_message(date_sent, marker, stripped_message_as_bytes, queue_url_cpa, xml, queue_is_fifo=False)


async def _process_messages(
    ibm_mq_client: IBMMQClient,
    queue_name: str,
    queue_url_clickout: str,
    queue_url_cpa: str,
    queue_url_boy: str,
):
    """Обработка сообщений из очереди Аэрофлота.

    :param ibm_mq_client: обёртка над клиентом для IBM MQ
    :param queue_url_clickout
    :param queue_url_cpa
    :param queue_url_boy
    :param queue_name: название очереди
    """
    with ibm_mq_client.queue_manager_context(settings.IBM_MQ_QUEUE_MANAGER) as queue_manager:
        with ibm_mq_client.queue_context(queue_manager, queue_name) as queue:
            while True:
                with ibm_mq_client.message_context(queue) as message:
                    if not message:
                        continue
                    await _handle_message(message, queue_url_clickout, queue_url_cpa, queue_url_boy)


def main():
    settings.setup_logging()

    if not settings.SQS_AEROFLOT_QUEUE_NAME:
        raise Exception('Empty AVIA_SQS_AEROFLOT_QUEUE_NAME')
    if not settings.SQS_AEROFLOT_BOY_QUEUE_NAME:
        raise Exception('Empty AVIA_SQS_AEROFLOT_BOY_QUEUE_NAME')

    logger.info('Creating queue "%s" in sqs ...', settings.SQS_AEROFLOT_QUEUE_NAME)
    queue_url_clickout = sqs_client.create_queue(settings.SQS_AEROFLOT_QUEUE_NAME)
    logger.info('Created: %s', queue_url_clickout)

    logger.info('Creating BoY queue "%s" in sqs ...', settings.SQS_AEROFLOT_BOY_QUEUE_NAME)
    queue_url_boy = sqs_client.create_queue(settings.SQS_AEROFLOT_BOY_QUEUE_NAME)
    logger.info('Created: %s', queue_url_boy)

    logger.info('Creating CPA collector queue "%s" in sqs ...', settings.SQS_AEROFLOT_CPA_QUEUE_NAME)
    queue_url_cpa = sqs_client.create_queue(settings.SQS_AEROFLOT_CPA_QUEUE_NAME, fifo=False)
    logger.info('Created: %s', queue_url_cpa)

    ibm_mq_client = IBMMQClient(
        host=settings.IBM_MQ_HOST,
        channel=settings.IBM_MQ_CHANNEL,
    )
    run(_process_messages(ibm_mq_client, settings.IBM_MQ_QUEUE_NAME, queue_url_clickout, queue_url_cpa, queue_url_boy))


if __name__ == '__main__':
    main()
