import logging
from contextlib import contextmanager
from typing import Optional, Iterator

import pymqi

logger = logging.getLogger(__name__)


class IBMMQClient(object):
    def __init__(
        self,
        host: str,
        channel: str,
        port: int = 1414,
        wait_interval: int = 5000,
    ):
        self._host = host
        self._port = port
        self._channel = channel
        self._message_descriptor = pymqi.MD()
        self._get_message_option = pymqi.GMO()
        self._get_message_option.Options = (
            pymqi.CMQC.MQGMO_WAIT
            | pymqi.CMQC.MQGMO_FAIL_IF_QUIESCING
            | pymqi.CMQC.MQGMO_BROWSE_FIRST
        )
        self._get_message_option.WaitInterval = wait_interval

        self._remove_message_option = pymqi.GMO()
        self._remove_message_option.Options = pymqi.CMQC.MQGMO_MSG_UNDER_CURSOR

    def _reset_message_descriptor(self):
        self._message_descriptor.MsgId = pymqi.CMQC.MQMI_NONE
        self._message_descriptor.CorrelId = pymqi.CMQC.MQCI_NONE
        self._message_descriptor.GroupId = pymqi.CMQC.MQGI_NONE

    @contextmanager
    def queue_manager_context(self, queue_manager_name: str) -> Iterator[pymqi.QueueManager]:
        """Контекстный менеджер для получения QueueManager.

        Дисконнектится при выходе из контекста.
        """
        conn_info = '{host}({port})'.format(host=self._host, port=self._port)

        logger.info('Trying to connect to %s at %s ...', queue_manager_name, conn_info)
        queue_manager = pymqi.connect(queue_manager_name, self._channel, conn_info)

        logger.info('Connected to %s at %s', queue_manager_name, conn_info)

        try:
            yield queue_manager
        finally:
            logger.info('Trying to disconnect from %s at %s ...', queue_manager_name, conn_info)
            queue_manager.disconnect()
            logger.info('Disconnected from %s at %s', queue_manager_name, conn_info)

    @contextmanager
    def queue_context(
        self, queue_manager: pymqi.QueueManager, queue_name: str,
    ) -> Iterator[pymqi.Queue]:
        """Контекстный менеджер для получения очереди.

        Закрывает очередь при выходе из контекста.
        """
        logger.info('Trying to open queue "%s"', queue_name)
        queue = pymqi.Queue(
            queue_manager, queue_name,
            (
                pymqi.CMQC.MQOO_FAIL_IF_QUIESCING
                | pymqi.CMQC.MQOO_INPUT_SHARED
                | pymqi.CMQC.MQOO_BROWSE
            ),
        )
        logger.info('Queue "%s" opened', queue_name)
        try:
            yield queue
        finally:
            logger.info('Closing queue "%s"', queue_name)
            queue.close()
            logger.info('Queue "%s" closed', queue_name)

    def _get_first_message(self, queue: pymqi.Queue) -> Optional[bytes]:
        """Получает верхнее сообщение из очереди, но не удаляет его.

        https://www.ibm.com/support/knowledgecenter/en/SSFKSJ_7.5.0/com.ibm.mq.dev.doc/q026470_.htm

        :param queue: объект очереди
        :return: полученное сообщение
        """
        logger.info('Waiting for a message')
        message = None
        while message is None:
            try:
                message = queue.get(None, self._message_descriptor, self._get_message_option)
                self._reset_message_descriptor()
                logger.info('Got a message')
                logger.debug(message)
                return message
            except pymqi.MQMIError as e:
                if e.comp == pymqi.CMQC.MQCC_FAILED and e.reason == pymqi.CMQC.MQRC_NO_MSG_AVAILABLE:
                    logger.debug('No messages')
                else:
                    logger.exception('Error: %s', e)
                    raise e

    def _remove_message_under_cursor(self, queue: pymqi.Queue) -> None:
        """Удалить из очереди сообщение, на которое указывает курсор.

        https://www.ibm.com/support/knowledgecenter/en/SSFKSJ_7.5.0/com.ibm.mq.dev.doc/q026490_.htm

        :param queue: объект очереди
        """
        try:
            logger.info('Trying to remove a message')
            queue.get(None, self._message_descriptor, self._remove_message_option)
            self._reset_message_descriptor()
        except pymqi.MQMIError as e:
            if e.comp == pymqi.CMQC.MQCC_FAILED and e.reason == pymqi.CMQC.MQRC_NO_MSG_AVAILABLE:
                logger.info('No message to remove')
            else:
                logger.exception('Error: %s', e)
                raise e
        else:
            logger.info('Removed the message successfully')

    @contextmanager
    def message_context(self, queue: pymqi.Queue) -> Iterator[Optional[bytes]]:
        """Контекстный менеджер для получения сообщений.

        При выходе из контекста удаляет сообщение, если не возникло ошибки.

        :param queue: объект очереди
        :return: контекстный менеджер, для использования с with
        """
        try:
            yield self._get_first_message(queue)
        except Exception:
            raise
        else:
            self._remove_message_under_cursor(queue)
