import logging

import boto3
import retrying

from travel.avia.aeroflot_queue_proxy import settings

logger = logging.getLogger(__name__)


class SQSClient(object):
    def __init__(self, endpoint_url, account):
        # type: (str, str) -> None
        self._client = boto3.client(
            'sqs',
            region_name='yandex',
            endpoint_url=endpoint_url,
            aws_access_key_id=account,
            aws_secret_access_key='',
        )

    @retrying.retry(
        stop_max_attempt_number=10,
        wait_exponential_multiplier=1000,
        wait_exponential_max=10000,
    )
    def create_queue(self, name, fifo=True, visibility_timeout='60'):
        # type: (str, bool, str) -> str
        attributes = {
            'VisibilityTimeout': str(visibility_timeout),
        }
        if fifo:
            attributes['FifoQueue'] = 'true'
        try:
            logger.info('Trying to create queue "%s" with attributes %r', name, attributes)
            response = self._client.create_queue(
                QueueName=name,
                Attributes=attributes,
            )
        except Exception:
            logger.exception('Error while creating queue "%s" with attributes %r', name, attributes)
            raise
        else:
            logger.info('Created queue "%s" with attributes %r successfully', name, attributes)
        return response['QueueUrl']

    @retrying.retry(
        stop_max_attempt_number=10,
        wait_exponential_multiplier=1000,
        wait_exponential_max=10000,
    )
    def send_message(self, **send_message_kwargs):
        response = self._client.send_message(
            **send_message_kwargs,
        )
        return response['MessageId']


sqs_client = SQSClient(settings.SQS_ENDPOINT, settings.SQS_ACCOUNT)
