#include <mapreduce/yt/interface/client.h>
#include <mapreduce/yt/interface/common.h>
#include <mapreduce/yt/common/config.h>

#include <library/cpp/getopt/last_getopt.h>

#include <travel/avia/analytics/price_changes/cpp/proto/avia_variant.pb.h>

#include <util/generic/vector.h>
#include <util/generic/ymath.h>
#include <util/stream/str.h>
#include <util/system/env.h>


namespace {
    void CopyVariantToAnswer(const NAvia::NProto::TAviaVariant& variant, NAvia::NProto::TAviaVariantWithAnswer& answer) {
        answer.SetFromSettlementId(variant.GetFromSettlementId());
        answer.SetFromAirportId(variant.GetFromAirportId());
        answer.SetToSettlementId(variant.GetToSettlementId());
        answer.SetToAirportId(variant.GetToAirportId());
        answer.SetForwardDate(variant.GetForwardDate());
        answer.SetBackwardDate(variant.GetBackwardDate());
        answer.SetAdults(variant.GetAdults());
        answer.SetChildren(variant.GetChildren());
        answer.SetInfants(variant.GetInfants());
        answer.SetClassId(variant.GetClassId());
        answer.SetPartnerId(variant.GetPartnerId());
        answer.SetVendorId(variant.GetVendorId());
        answer.SetUnixtime(variant.GetUnixtime());
        answer.SetForwardRoute(variant.GetForwardRoute());
        answer.SetBackwardRote(variant.GetBackwardRote());
        answer.SetWithBaggage(variant.GetWithBaggage());
        answer.SetOriginalPrice(variant.GetOriginalPrice());
        answer.SetTimeToDepart(variant.GetTimeToDepart());
        answer.SetDaysToFlight(variant.GetDaysToFlight());
    }

    const NYT::TSortColumns KeyColumns = {
        "from_settlement_id",
        "from_airport_id",
        "to_settlement_id",
        "to_airport_id",
        "forward_date",
        "backward_date",
        "adults",
        "children",
        "infants",
        "class_id",

        "forward_route",
        "backward_rote",

        "partner_id",
        "vendor_id"

    };

    NYT::TSortColumns GetSortColumns() {
        NYT::TSortColumns base = KeyColumns;
        base.Add("unixtime");
        return base;
    }
}


class TAnswerAppenderReducer : public NYT::IReducer<
    NYT::TTableReader<NAvia::NProto::TAviaVariant>,
    NYT::TTableWriter<NAvia::NProto::TAviaVariantWithAnswer>>{
public:
    using TInputType = NAvia::NProto::TAviaVariant;
    using TOutputType = NAvia::NProto::TAviaVariantWithAnswer;
    TAnswerAppenderReducer() = default;
    TAnswerAppenderReducer(i64 threshold)
        : threshold(threshold)
        {
        }

    Y_SAVELOAD_JOB(threshold);

    void Do(TReader* reader, TWriter* writer) override {
        TVector<TInputType> records;
        for (; reader->IsValid(); reader->Next()) {
            records.push_back(std::move(reader->GetRow()));
        }

        TOutputType answerRecord;
        for (size_t i = 0; i < records.size(); ++i) {
            auto& record = records[i];
            i64 currentPrice = static_cast<i64>(record.GetOriginalPrice());

            CopyVariantToAnswer(record, answerRecord);
            answerRecord.ClearAnswer();
            for (size_t j = i + 1; j < records.size(); ++j) {
                i64 newPrice = static_cast<i64>(records[j].GetOriginalPrice());
                i64 delta = newPrice - currentPrice;
                if (Abs(delta) > threshold) {
                    answerRecord.SetAnswer(records[j].GetUnixtime() - record.GetUnixtime());
                    break;
                }
            }
            writer->AddRow(answerRecord);
        }
    }


private:
    i64 threshold;
};


struct TCommandLineArguments {
    TString Proxy;
    TString InputTable;
    TString OutputTable;
    i64 Threshold;

    static TCommandLineArguments Parse(int argc, const char** argv);
};

TCommandLineArguments TCommandLineArguments::Parse(int argc, const char * argv[]) {
    TCommandLineArguments result;

    result.Proxy = GetEnv("YT_PROXY", "hahn");

    auto opts = NLastGetopt::TOpts::Default();
    opts.AddLongOption("src-table").StoreResult(&result.InputTable).Required().Help("Source table");
    opts.AddLongOption("dst-table").StoreResult(&result.OutputTable).Required().Help("Destination table");
    opts.AddLongOption("threshold").StoreResult(&result.Threshold).Required().Help("Minimal price threshold");

    NLastGetopt::TOptsParseResult args(&opts, argc, argv);
    return result;
}


REGISTER_REDUCER(TAnswerAppenderReducer);

int main(int argc, const char** argv)
{
    NYT::TConfig::Get()->InferTableSchema = true;
    NYT::TConfig::Get()->LogLevel = "INFO";

    NYT::Initialize(argc, argv);

    TCommandLineArguments args = TCommandLineArguments::Parse(argc, argv);

    auto client = NYT::CreateClient(args.Proxy);

    client->MapReduce(
        NYT::TMapReduceOperationSpec()
        .AddInput<NAvia::NProto::TAviaVariant>(args.InputTable)
        .AddOutput<NAvia::NProto::TAviaVariantWithAnswer>(args.OutputTable)
        .ReduceBy(KeyColumns)
        .SortBy(GetSortColumns()),
        nullptr,
        new TAnswerAppenderReducer(args.Threshold)
    );
}
