from collections import namedtuple

LandingRoute = namedtuple(
    'LandingRoute',
    (
        'from_id',
        'to_id',
        'national_version',
    ),
)


class LandingRoutesCache:
    def __init__(self, logger):
        self.landing_routes = set()
        self.logger = logger

    def __contains__(self, landing_route):
        # type: (LandingRoute) -> bool
        return landing_route in self.landing_routes

    def populate(self):
        self.landing_routes = self.build_cache()
        self.logger.info('populated cache with %d landing routes', len(self.landing_routes))

    def update_cache(self):
        try:
            self.landing_routes = self.build_cache()
        except Exception as e:
            self.logger.error('Failed to update landing routes cache. Will continue using old cache. Reason: %s', e)

        self.logger.info('repopulated cache with %d landing routes', len(self.landing_routes))

    @classmethod
    def build_cache(cls):
        raw_routes = cls._load_routes()
        routes = set()
        for r in raw_routes:
            routes.add(LandingRoute(r['from_id'], r['to_id'], r['national_version']))

        return routes

    @staticmethod
    def _load_routes():
        from travel.avia.api_gateway.application.fetcher.avia_statistics import LandingRoutesFetcher

        fetcher = LandingRoutesFetcher()
        return fetcher.fetch_sync()
