# -*- encoding: utf-8 -*-
import logging
from collections import namedtuple
from datetime import date, timedelta
from functools import partial
from typing import Optional

from pydantic import BaseModel, Field
from tornado.web import HTTPError
import tornado.escape

from travel.proto.dicts.rasp.settlement_pb2 import TSettlement

from travel.avia.api_gateway.application.fetcher import Fetcher
from travel.avia.api_gateway.application.fetcher.anywhere_landing.memcache import AnywhereLandingCache
from travel.avia.api_gateway.application.fetcher.avatars import SettlementImagesFetcher
from travel.avia.api_gateway.application.fetcher.country.types import (
    EKlass,
    FromPoint,
    SettlementInfo,
)
from travel.avia.api_gateway.application.fetcher.country_restrictions import CountryRestrictionsDataFetcher
from travel.avia.api_gateway.application.fetcher.price_index import MinPriceBatchSearch
from travel.avia.library.python.enum import NationalVersion, Language
from travel.avia.api_gateway.application.fetcher.country.fetcher import (
    linguistics_for_point,
    on_prices_and_images,
)
from travel.avia.api_gateway.lib.model_utils import get_settlement_by_code, get_settlement_id, get_point_key


logger = logging.getLogger(__name__)


class AnywhereLandingResponse(BaseModel):
    from_point: FromPoint = Field(alias='fromPoint')
    settlements: list[SettlementInfo]

    class Config:
        allow_population_by_field_name = True


class AnywhereLandingRequest(BaseModel):
    from_point: str = Field(alias='from')
    date_forward: Optional[date] = Field(alias='dateForward')
    date_backward: Optional[date] = Field(alias='dateBackward')
    adults: int = 1
    children: int = 0
    infants: int = 0
    klass: EKlass = EKlass.economy
    nv: Optional[NationalVersion] = NationalVersion.RU
    lang: Optional[Language] = Language.RU
    limit: int = 250


# Список городов которые отдаем, пока фиксированный
Destination = namedtuple('Destination', ['id', 'popularity'])


DESTINATION_CITIES = [
    Destination(213, 1590442439),   # Москва
    Destination(2, 558996710),   # Санкт-Петербург
    Destination(239, 452412114),   # Сочи
    Destination(54, 180503585),   # Екатеринбург
    Destination(146, 177892664),   # Симферополь
    Destination(65, 165040049),   # Новосибирск
    Destination(11063, 136458648),   # Минеральные Воды
    Destination(35, 134352753),   # Краснодар
    Destination(22, 127702623),   # Калининград
    Destination(43, 115910346),   # Казань
    Destination(172, 114892023),   # Уфа
    Destination(62, 91557628),   # Красноярск
    Destination(51, 87547140),   # Самара
    Destination(1107, 85765452),   # Анапа
    Destination(39, 70923181),   # Ростов-на-Дону
    Destination(56, 70839282),   # Челябинск
    Destination(55, 69538880),   # Тюмень
    Destination(28, 67450296),   # Махачкала
    Destination(63, 64131926),   # Иркутск
    Destination(50, 61127239),   # Пермь
    Destination(66, 59230880),   # Омск
    Destination(23243, 58557776),   # Нижний Новгород
    Destination(75, 51495447),   # Владивосток
    Destination(973, 46444170),   # Сургут
    Destination(38, 44351203),   # Волгоград
    Destination(10990, 34157871),   # Геленджик
    Destination(76, 32740543),   # Хабаровск
    Destination(194, 31035078),   # Саратов
    Destination(23, 30531703),   # Мурманск
    Destination(197, 29109562),   # Барнаул
    Destination(48, 25679688),   # Оренбург
    Destination(37, 24603089),   # Астрахань
    Destination(193, 23232485),   # Воронеж
    Destination(20, 23142908),   # Архангельск
    Destination(64, 20708187),   # Кемерово
    Destination(33, 20284380),   # Владикавказ
    Destination(1091, 19818612),   # Нижневартовск
    Destination(36, 19773967),   # Ставрополь
    Destination(67, 19635905),   # Томск
    Destination(23242, 18995248),   # Южно-Сахалинск
    Destination(237, 17758638),   # Новокузнецк
    Destination(78, 17547225),   # Петропавловск-Камчатский
    Destination(11230, 17512391),   # Новый Уренгой
    Destination(19, 16684523),   # Сыктывкар
    Destination(44, 16653851),   # Ижевск
    Destination(195, 16160176),   # Ульяновск
    Destination(198, 15965877),   # Улан-Удэ
    Destination(1106, 15453312),   # Грозный
    Destination(77, 14053184),   # Благовещенск
    Destination(46, 14028309),   # Киров
    Destination(11508, 153684280),   # Стамбул
    Destination(10262, 108170498),   # Ереван
    Destination(11511, 89961294),   # Анталья
    Destination(10335, 79445327),   # Ташкент
    Destination(11499, 51869943),   # Дубай
    Destination(10253, 51742744),   # Баку
    Destination(10318, 37719167),   # Душанбе
    Destination(10310, 37361853),   # Ош
    Destination(10309, 34806179),   # Бишкек
    Destination(157, 33751204),   # Минск
    Destination(22177, 27354784),   # Алматы
    Destination(163, 26273154),   # Нур-Султан (Астана)
    Destination(131, 25114182),   # Тель-Авив
    Destination(10322, 22995862),   # Худжанд
    Destination(10313, 20528333),   # Кишинёв
    Destination(10522, 20277626),   # Белград
    Destination(11487, 19791998),   # Шарм-эль-Шейх
    Destination(10334, 17096927),   # Самарканд
    Destination(10421, 16527185),   # Ларнака
    Destination(11486, 16390469),   # Хургада
    Destination(10398, 15989846),   # Будапешт
    Destination(10277, 15788257),   # Тбилиси
    Destination(10622, 14535208),   # Пхукет
    Destination(10605, 13760276),   # Мале
    Destination(10502, 13264075),   # Париж
    Destination(21105, 12651683),   # Ургенч
    Destination(10330, 12067640),   # Бухара
    Destination(10336, 11816997),   # Фергана
    Destination(10393, 11535761),   # Лондон
    Destination(202, 11287767),   # Нью-Йорк
    Destination(10418, 10457594),   # Афины
    Destination(21314, 9807754),   # Наманган
    Destination(10620, 9501291),   # Бангкок
    Destination(100, 9160923),   # Франкфурт-на-Майне
    Destination(10633, 9027728),   # Коломбо
    Destination(11485, 8350085),   # Каир
    Destination(10445, 8213276),   # Рим
    Destination(10429, 8193722),   # Барселона
    Destination(21420, 8098021),   # Канкун
    Destination(10472, 7916628),   # Варшава
    Destination(177, 7648379),   # Берлин
    Destination(10380, 7578525),   # Бургас
    Destination(21445, 7508368),   # Dalaman
    Destination(11498, 7212081),   # Абу-Даби
    Destination(10511, 7117219),   # Прага
    Destination(10635, 7050483),   # Сеул
    Destination(10466, 6899972),   # Амстердам
    Destination(10448, 6755484),   # Милан
    Destination(10371, 6234198),   # Вена
    Destination(10493, 6222186),   # Хельсинки
    Destination(22245, 5926060),   # Куляб
    Destination(99, 5631528),   # Мюнхен
    Destination(10259, 5619685),   # Гюмри
    Destination(10278, 5528709),   # Батуми
    Destination(10379, 5439395),   # София
    Destination(22271, 5428106),   # Пунта Кана
    Destination(29961, 5363484),   # Газипаша
    Destination(22187, 5278884),   # Лос-Анджелес
    Destination(10424, 5006632),   # Салоники
    Destination(10562, 4946734),   # Дели
    Destination(11474, 4898343),   # Рига
    Destination(10636, 4568046),   # Токио
    Destination(22176, 4399917),   # Шымкент
    Destination(10331, 4393043),   # Карши
    Destination(22238, 4130597),   # Тиват
    Destination(10435, 4012125),   # Мадрид
    Destination(23370, 4004955),   # Майами
    Destination(10286, 3985810),   # Актау
    Destination(21242, 3932356),   # Пафос
    Destination(21487, 3907109),   # Доха
    Destination(10381, 3865135),   # Варна
    Destination(10408, 3729448),   # Дюссельдорф
    Destination(11513, 3708923),   # Денпасар
    Destination(10507, 3689791),   # Загреб
    Destination(21611, 3586530),   # Подгорица
    Destination(11503, 3469617),   # Ankara
    Destination(20272, 3394698),   # Мехико
    Destination(20843, 3315293),   # Шарджа
    Destination(10515, 3187795),   # Цюрих
    Destination(11481, 3100415),   # Таллин
    Destination(10478, 3052121),   # Лиссабон
    Destination(10338, 2909000),   # Термез
    Destination(10590, 2892183),   # Пекин
    Destination(10519, 2725244),   # Стокгольм
    Destination(20835, 2677382),   # Аликанте
    Destination(10419, 2641274),   # Ираклион
    Destination(10329, 2591324),   # Андижан
    Destination(26812, 2591049),   # Гоа
    Destination(10376, 2587175),   # Брюссель
    Destination(10500, 2577719),   # Ницца
    Destination(11505, 2574141),   # İzmir
    Destination(20273, 2501931),   # Актобе
    Destination(10141, 2481171),   # Виктория
    Destination(10115, 2457398),   # Торонто
    Destination(11475, 2405883),   # Вильнюс
    Destination(11501, 2404433),   # Adana
    Destination(10137, 2337111),   # Гавана
    Destination(10467, 2312208),   # Осло
    Destination(178, 2309164),   # Гамбург
    Destination(10514, 2277529),   # Женева
    Destination(10306, 2277466),   # Усть-Каменогорск
    Destination(10319, 2264817),   # Бохтар
    Destination(10487, 2230572),   # Бухарест
    Destination(10451, 2195664),   # Венеция
    Destination(10291, 2184727),   # Атырау
    Destination(87, 2137912),   # Вашингтон
    Destination(10441, 2065007),   # Тенерифе
    Destination(10619, 1972345),   # Сингапур
    Destination(10138, 1958296),   # Касабланка
    Destination(10131, 1956275),   # Чикаго
    Destination(10425, 1930943),   # Копенгаген
    Destination(10254, 1925321),   # Гянджа
    Destination(10437, 1925150),   # Малага
    Destination(11495, 1905080),   # Тегеран
    Destination(10552, 1863555),   # Ханой
    Destination(10374, 1829027),   # Тирана
    Destination(10599, 1803046),   # Шанхай
    Destination(10133, 1785363),   # Буэнос-Айрес
    Destination(98, 1752541),   # Кельн
    Destination(90, 1701443),   # Сан-Франциско
    Destination(10296, 1664162),   # Навои
    Destination(10492, 1659858),   # Любляна
    Destination(10143, 1644178),   # Тунис
    Destination(10423, 1644178),   # Родос
    Destination(10145, 1630755),   # Сидней
    Destination(21354, 1598557),   # Варадеро
    Destination(30026, 1561795),   # Muğla
    Destination(20924, 1547640),   # Сан-Паулу
    Destination(10426, 1504782),   # Дублин
    Destination(10305, 1497351),   # Уральск
    Destination(21221, 1476573),   # Рио-Де-Жанейро
    Destination(10446, 1465884),   # Неаполь
    Destination(10553, 1451969),   # Хошимин
    Destination(101, 1445236),   # Штутгарт
    Destination(11514, 1428695),   # Гонконг
    Destination(11494, 1420402),   # Амман
    Destination(10430, 1404400),   # Валенсия
    Destination(10279, 1370865),   # Кутаиси
    Destination(10574, 1339753),   # Джакарта
    Destination(40246, 1306199),   # Нячанг
    Destination(10444, 1298226),   # Болонья
    Destination(10506, 1284290),   # Дубровник
    Destination(20778, 1281871),   # Самуи
    Destination(26600, 1264431),   # Kayseri
    Destination(11497, 1261858),   # Бейрут
    Destination(10337, 1241485),   # Нукус
    Destination(223, 1232025),   # Бостон
    Destination(22256, 1231229),   # Керкира
    Destination(22320, 913616),   # Фуншал
    Destination(22264, 866609),   # Мальта
    Destination(20769, 277404),   # Эль-Сальвадор
]


class AnywhereLandingFetcher(Fetcher):
    def __init__(self, anywhere_request: AnywhereLandingRequest, *args, **kwargs):
        super(AnywhereLandingFetcher, self).__init__(*args, **kwargs)
        self.anywhere_request: AnywhereLandingRequest = anywhere_request

    def fetch(self, fetchers=None):
        try:
            point_from = get_settlement_by_code(self.cache_root, self.anywhere_request.from_point)
        except BaseException:
            raise HTTPError(404, reason=f'Unknown point {tornado.escape.url_escape(self.anywhere_request.from_point)}'[:40])

        if point_from is None:
            raise HTTPError(404, reason=f'Unknown point {tornado.escape.url_escape(self.anywhere_request.from_point)}'[:40])

        from_id = get_settlement_id(point_from)
        cities = [city for city in DESTINATION_CITIES if not city.id == from_id]
        cities = cities[:self.anywhere_request.limit]

        base_direction_request = {
            'forward_date': str(self.anywhere_request.date_forward) if self.anywhere_request.date_forward else
            (date.today() + timedelta(days=1)).strftime('%Y-%m-%d'),
            'backward_date': str(self.anywhere_request.date_backward) if self.anywhere_request.date_backward else None,
            'from_id': from_id,
            'adults_count': self.anywhere_request.adults,
            'children_count': self.anywhere_request.children,
            'infants_count': self.anywhere_request.infants,
        }
        fetcher = Fetcher(
            finish_callback=partial(self.on_prices_and_images, point_from, cities),
        )
        fetcher.waiting_fields = {'prices', 'country_restrictions'}
        fetcher.fetch(
            [
                MinPriceBatchSearch(
                    field='prices',
                    national_version=self.anywhere_request.nv,
                    request={'min_requests': [dict(base_direction_request, to_id=city.id) for city in cities]},
                ),
                CountryRestrictionsDataFetcher(
                    request_headers=[],
                    field='country_restrictions',
                    from_id=from_id,
                    to_ids=[city.id for city in cities if not city.id == from_id],
                ),
            ],
        )

    def on_prices_and_images(
        self,
        point_from: TSettlement,
        cities: list[Destination],
        prices_and_images,
    ):
        city_ids = [
            {
                'cityId': city.id,
                'popularity': city.popularity,
            } for city in cities
        ]
        prices_and_images['images'] = AnywhereLandingCache.get_images_for_all_settlements()
        settlements = on_prices_and_images(self.cache_root, city_ids, prices_and_images, self.anywhere_request.lang)
        self.finish_callback(
            AnywhereLandingResponse(
                from_point=FromPoint(
                    key=get_point_key(point_from),
                    title=linguistics_for_point(point_from, self.anywhere_request.lang),
                ),
                settlements=settlements,
            ).dict(by_alias=True)
        )


def init_settlements_images_cache():
    for city in DESTINATION_CITIES:
        try:
            fetcher = Fetcher(finish_callback=partial(_cache_images_for_city, city.id))
            fetcher.waiting_fields = {'images'}
            fetcher.fetch([
                SettlementImagesFetcher(
                    make_request_sync=True,
                    field='images',
                    settlement_id=city.id,
                ),
            ])
        except HTTPError:
            logger.warn('unable to cache image for city %d', city.id)

    images_for_all_cities = {}
    for city in DESTINATION_CITIES:
        city_images = AnywhereLandingCache.get_images_for_settlement(city.id)
        if city_images:
            images_for_all_cities[city.id] = city_images
    AnywhereLandingCache.set_images_for_all_settlements(images_for_all_cities)
    logger.info('cached images for the %d settlements', len(images_for_all_cities))


def _cache_images_for_city(city_id: int, result: dict):
    if not result.get('images'):
        logger.warn('unable to cache image for the city %d', city_id)
    else:
        AnywhereLandingCache.set_images_for_settlement(city_id, result['images'])
