from enum import Enum
from functools import partial
from typing import Optional, List
import logging

from tornado.httpclient import HTTPResponse
import tornado.escape

from travel.avia.api_gateway import settings
from travel.avia.api_gateway.application.fetcher import Fetcher
from travel.avia.api_gateway.application.fetcher.content_backend import ContentBackendCityImageFetcher
from travel.avia.api_gateway.lib.avatars.url import AvatarsInstallationUrl, AvatarsUrlDescriptor, Operation
from travel.avia.api_gateway.lib.avatars.sizes import AvatarSize, AVATARS_SIZES_BY_NAMESPACE

AVATARS_GET_IMAGE_INFO_CONNECT_TIMEOUT = 2.0
AVATARS_GET_IMAGE_INFO_REQUEST_TIMEOUT = 2.0

logger = logging.getLogger(__name__)


class AvatarsBaseFetcher(Fetcher):
    pass


class AvatarsParameters(str, Enum):
    scheme = 'scheme'
    netloc = 'netloc'
    namespace = 'namespace'
    group_id = 'group_id'
    image_name = 'image_name'


class AvatarsGetImageInfoFetcher(Fetcher):
    """
    Request avatars internal api for meta information on image
    """

    service = 'avatars_get_image_info'

    _cache_ = {}

    @classmethod
    def _cache_key_(cls, netloc, scheme, namespace, group_id, image_name):
        return netloc, scheme, namespace, group_id, image_name

    def fetch(self, fetchers: Optional[List['Fetcher']] = None) -> None:
        netloc: str = self.params[AvatarsParameters.netloc]
        scheme: str = self.params[AvatarsParameters.scheme]
        namespace: str = self.params[AvatarsParameters.namespace]
        group_id: str = self.params[AvatarsParameters.group_id]
        image_name: str = self.params[AvatarsParameters.image_name]

        cache_key = AvatarsGetImageInfoFetcher._cache_key_(
            netloc,
            scheme,
            namespace,
            group_id,
            image_name,
        )
        if cache_key in AvatarsGetImageInfoFetcher._cache_:
            self.finish_callback(AvatarsGetImageInfoFetcher._cache_[cache_key], self.field)
            return

        url: str = AvatarsUrlDescriptor(
            installation=AvatarsInstallationUrl(scheme=scheme, netloc=netloc),
            operation=Operation.getimageinfo,
            namespace=namespace,
            group_id=group_id,
            image_name=image_name,
        ).url()

        logger.info(f'request {url=}')
        on_response = partial(self.on_result, cache_key)
        self.request(url, callback=on_response)

    def on_result(self, cache_key, response: HTTPResponse):
        data = tornado.escape.json_decode(response.body)
        AvatarsGetImageInfoFetcher._cache_[cache_key] = data
        self.finish_callback(data, self.field)


class SettlementImagesFetcher(Fetcher):
    """
    Request images for single settlement id.
    Result is list of lists of dicts list[list[dict]], that describe different images (outer list) with different sizes (inner list)
    """
    def __init__(self, make_request_sync: bool = False, *args, **kwargs):
        super(SettlementImagesFetcher, self).__init__(*args, **kwargs)
        self._make_request_sync = make_request_sync

    def fetch(self, fetchers: Optional[List['Fetcher']] = None) -> None:
        settlement_id = self.params.get('settlement_id')
        fetcher = ContentBackendCityImageFetcher(
            make_request_sync=self._make_request_sync,
            finish_callback=self.on_content_backend_response,
            content_backend_api_url=settings.CONTENT_BACKEND_API_URL,
            settlement_id=settlement_id,
        )
        fetcher.fetch()

    def on_content_backend_response(self, response: dict, field: str):
        if not response:
            return self.finish_callback([], field=self.field)
        original_image = response['image_url']
        request_descriptor: AvatarsUrlDescriptor = AvatarsUrlDescriptor.from_orig_image_url(
            original_image, Operation.getimageinfo
        )
        if request_descriptor.namespace not in AVATARS_SIZES_BY_NAMESPACE:
            return self.finish_callback([], field=self.field)

        if request_descriptor.namespace in AVATARS_SIZES_BY_NAMESPACE:
            # use known sizes for namespace
            get_image_descriptor = AvatarsUrlDescriptor.from_orig_image_url(original_image, Operation.get)
            sizes = []
            for image_size in AVATARS_SIZES_BY_NAMESPACE[request_descriptor.namespace]:
                image_size: AvatarSize
                sizes.append(
                    {
                        'url': get_image_descriptor.url_for_image_alias(image_size.alias),
                        'width': image_size.width,
                        'height': image_size.height,
                    }
                )
            return self.finish_callback([sizes], field=self.field)

    def on_result(self, original_image: str, data):
        get_image_descriptor = AvatarsUrlDescriptor.from_orig_image_url(original_image, Operation.get)
        sizes = []
        for image_data in data['sizes'].values():
            image_data['url'] = get_image_descriptor.url_for_path(image_data['path'])
            sizes.append(image_data)
        self.finish_callback([sizes], field=self.field)


class SettlementsImagesFetcher(Fetcher):
    """
    Request images for a set of settlement ids
    Result is a dict with settlement id as key and response from SettlementImagesFetcher as value
    """

    def fetch(self, fetchers: Optional[List['Fetcher']] = None) -> None:
        settlement_ids = self.params.get('settlement_ids')

        fetchers = []
        fields = set()
        for settlement_id in settlement_ids:
            fetchers.append(
                SettlementImagesFetcher(
                    cache_root=self.cache_root,
                    settlement_id=settlement_id,
                    field=settlement_id,
                )
            )
            fields.add(settlement_id)

        fetcher = Fetcher(finish_callback=self.on_result)
        fetcher.waiting_fields = fields
        fetcher.fetch(fetchers)

    def on_result(self, data):
        self.finish_callback(data, field=self.field)
