# -*- encoding: utf-8 -*-
from collections import defaultdict
from typing import Optional
from urllib.parse import urlencode

import tornado.escape
from tornado.httpclient import HTTPResponse
from tornado.web import MissingArgumentError

from travel.avia.api_gateway.application.fetcher import Fetcher
from travel.avia.api_gateway.settings import BACKEND_API_URL


class BaseFetcher(Fetcher):
    service = 'avia_backend'

    def do_fetch(self, method, params=None, fields=None, query=()):
        # type: (str, dict, list, Optional[dict]) -> None
        query = dict(query, name=method)
        if self.params.get('lang'):
            query['lang'] = self.params['lang']
        self.request(
            '{}?{}'.format(BACKEND_API_URL, urlencode(query)),
            method='POST',
            headers={
                'Content-Type': 'application/json',
            },
            body=tornado.escape.json_encode(
                [
                    {
                        'name': method,
                        'params': params,
                        'fields': fields,
                    }
                ]
            ),
            callback=self.on_response,
        )

    def do_fetch_rest(self, uri, params=None):
        # type: (str, Optional[dict]) -> None
        self.request(
            '{}/rest{}'.format(BACKEND_API_URL, uri),
            method='GET',
            headers={
                'Content-Type': 'application/json',
            },
            params=params,
            callback=self.on_response_rest,
        )

    def on_response(self, response):
        # type: (HTTPResponse) -> None
        data = tornado.escape.json_decode(response.body)
        self.finish_callback(data['data'][0], field=self.field)

    def on_response_rest(self, response):
        # type: (HTTPResponse) -> None
        data = tornado.escape.json_decode(response.body)
        self.finish_callback(data['data'], field=self.field)


class AirportsFetcher(BaseFetcher):
    service = 'avia_backend_airports'

    def fetch(self, fetchers=None):
        airports = self.params.get('airports', {})
        if not airports:
            raise Exception('Empty "airports"')

        self.do_fetch_rest('/airports', {'id': ','.join(map(str, airports.values())), 'lang': self.params.get('lang')})

    def on_response_rest(self, response):
        # type: (HTTPResponse) -> None
        airports = self.params.get('airports', {})
        fields = defaultdict(set)
        for field, airport_id in airports.items():
            fields[airport_id].add(field)

        data = tornado.escape.json_decode(response.body)
        fetched_ids = set()
        for airport in data['data']:
            if airport['id'] in fetched_ids:
                continue
            fetched_ids.add(airport['id'])
            for f in fields[airport['id']]:
                self.finish_callback(airport, field=f)
        for airport_id in airports.values():
            if airport_id not in fetched_ids:
                for f in fields[airport_id]:
                    self.finish_callback(None, field=f)


class StationFetcher(BaseFetcher):
    service = 'avia_backend_station'

    def fetch(self, fetchers=None):
        station_id = self.params.get('station_id')
        if not station_id:
            raise Exception('Empty "station_id"')

        self.do_fetch(
            'station',
            params={'id': station_id},
            fields=[
                'id',
                'title',
                'url_title',
                'key',
                'code',
                'iata_code',
                'geo_id',
                'iata_code',
                'popular_title',
                'phrase_to',
                'phrase_from',
                'phrase_in',
                'station_type',
                {
                    'fields': [
                        'title',
                        'geo_id',
                        {
                            'fields': [
                                'title',
                                'geo_id',
                            ],
                            'name': 'country',
                        },
                    ],
                    'name': 'settlement',
                },
                'longitude',
                'latitude',
                'time_zone',
                'time_zone_utc_offset',
            ],
        )


class StationsFetcher(BaseFetcher):
    service = 'avia_backend_stations'

    def fetch(self, fetchers=None):
        settlement_id = self.params.get('settlement_id')
        if not settlement_id:
            raise Exception('Empty "settlement_id"')

        self.do_fetch(
            'stations',
            params={'settlementId': settlement_id, 'ttype': 'airport'},
            fields=[
                'id',
                'title',
                'url_title',
                'key',
                'code',
                'iata_code',
                'geo_id',
                'iata_code',
                'popular_title',
                'phrase_to',
                'phrase_from',
                'phrase_in',
                'station_type',
                {
                    'fields': [
                        'title',
                        'geo_id',
                        {
                            'fields': [
                                'title',
                                'geo_id',
                            ],
                            'name': 'country',
                        },
                    ],
                    'name': 'settlement',
                },
                'longitude',
                'latitude',
                'time_zone',
                'time_zone_utc_offset',
            ],
        )


class GeoPointFetcher(BaseFetcher):
    service = 'avia_backend_geo_point'

    def fetch(self, fetchers=None):
        if not self.params.get('key'):
            raise MissingArgumentError('key')

        self.do_fetch_rest(
            '/geo/point',
            params={
                'key': self.params.get('key'),
                'lang': self.params.get('lang'),
            },
        )


class AirlineFetcher(BaseFetcher):
    service = 'avia_backend_airline'

    def fetch(self, fetchers=None):
        airline_id = self.params.get('airline_id')
        fields = self.params.get('fields', 'id,alliance,title,url,logoSvg,color,slug,iata,sirena')
        if not airline_id:
            raise Exception('Empty "airline_id"')

        self.do_fetch_rest(
            '/airlines/airline_info/{}'.format(airline_id),
            params={
                'lang': self.params.get('lang', 'ru'),
                'fields': fields,
            },
        )


class PartnersPopularByRouteFetcher(BaseFetcher):
    service = 'avia_backend_partners_popular_by_route'

    def fetch(self, fetchers=None):
        self.do_fetch_rest(
            '/partners/popular_by_route/{national_version}/{lang}/{from_point_key}/{to_point_key}'.format(
                national_version=self.params.get('national_version'),
                lang=self.params.get('lang', 'ru'),
                from_point_key=self.params.get('from_point_key'),
                to_point_key=self.params.get('to_point_key'),
            )
        )


class ReviewsFetcher(BaseFetcher):
    service = 'avia_backend_reviews'

    def fetch(self, fetchers=None):
        airline_id = self.params.get('airline_id')
        flight_number = self.params.get('flight_number')
        if not airline_id:
            raise Exception('Empty "airline_id"')
        if not flight_number:
            raise Exception('Empty "flight_number"')

        self.do_fetch_rest('/airlines/{}/flights/{}/reviews'.format(airline_id, flight_number))


class NearDistancesFetcher(BaseFetcher):
    service = 'avia_backend_near_distances'

    def fetch(self, fetchers=None):
        params = {
            'fromId': self.params['from_id'],
            'toId': self.params['to_id'],
        }

        self.do_fetch('nearDistances', params=params)


class NearDirectionsFetcher(BaseFetcher):
    service = 'avia_backend_near_directions'

    def fetch(self, fetchers=None):
        params = {
            'fromId': self.params['from_id'],
            'toId': self.params['to_id'],
            'distance': self.params['distance'],
            'when': self.params['forward_date'],
            'return_date': self.params.get('backward_date'),
            'adultSeats': self.params['adult_seats'],
            'childrenSeats': self.params['children_seats'],
            'infantSeats': self.params['infant_seats'],
            'klass': self.params['klass'],
        }
        fields = [
            {'name': 'fromCity', 'fields': ['key', 'title', 'country', 'id']},
            {
                'name': 'toCity',
                'fields': ['key', 'title', 'geoId', 'country', 'latitude', 'longitude', 'iataCode', 'id', 'urlTitle'],
            },
        ]
        self.do_fetch('nearDirections', params=params, fields=fields)


class AirlinesSynonymsFetcher(BaseFetcher):
    service = 'avia_backend_airlines_synonyms'

    def fetch(self, fetchers=None):
        # type: (Optional[list]) -> None

        self.request(
            '{}/airlines/synonyms'.format(BACKEND_API_URL),
            callback=self.on_response,
        )

    def on_response(self, response):
        # type: (HTTPResponse) -> None
        data = tornado.escape.json_decode(response.body)
        self.finish_callback(data, field=self.field)


class FlightRatingFetcher(BaseFetcher):
    service = 'avia_backend_flight_rating'

    def fetch(self, fetchers=None):
        flight_number = self.params.get('flight_number')
        if not flight_number:
            raise Exception('Empty "flight_number"')

        self.do_fetch_rest('/flights/flight_rating/{}'.format(flight_number))


class TravelRecipesFetcher(BaseFetcher):
    service = 'avia_backend_travel_recipes'

    def fetch(self, fetchers=None):
        if not self.params.get('from_geo_id'):
            raise MissingArgumentError('from_geo_id')

        if not self.params.get('national_version'):
            raise MissingArgumentError('national_version')

        self.do_fetch_rest('/travel-recipes', params=self.params)


class AirportTabloSourceFetcher(BaseFetcher):
    service = 'avia_backend_airport_tablo_source'

    def fetch(self, fetchers=None):
        params = {}
        trusted = self.params.get('trusted')
        if trusted is not None:
            params['trusted'] = trusted

        self.do_fetch_rest('/airport-tablo-source', params)


class TopFlightsFetcher(BaseFetcher):
    service = 'avia_backend_top_flights'

    def fetch(self, fetchers=None):
        query = {}
        national_version = self.params.get('national_version')
        if national_version:
            query['national_version'] = national_version

        self.do_fetch(
            'topFlights',
            params={
                'fromKey': self.params.get('point_from'),
                'toKey': self.params.get('point_to'),
                'date': self.params.get('date'),
                'limit': self.params.get('limit'),
            },
            query=query,
        )


class CurrencyFetcher(BaseFetcher):
    service = 'avia_backend_currency'

    def fetch(self, fetchers=None):
        query = {}
        national_version = self.params.get('national_version')
        if national_version:
            query['national_version'] = national_version

        self.do_fetch('currency', query=query)
