# -*- encoding: utf-8 -*-
import logging
from typing import Dict

from tornado.web import HTTPError

from travel.avia.api_gateway.application.fetcher import Fetcher
from travel.avia.api_gateway.application.fetcher.city_to_landing.mapper import CityToLandingMapper
from travel.avia.api_gateway.application.fetcher.avia_statistics import CityToLandingDataFetcher

logger = logging.getLogger(__name__)


class CityToLandingFetcher(Fetcher):
    service = 'city_to_landing'

    class _FIELDS:
        statistics = 'statistics'

    def __init__(self, city_to_landing_mapper, *args, **kwargs):
        # type: (CityToLandingMapper, any, any) -> None
        super(CityToLandingFetcher, self).__init__(*args, **kwargs)
        self._city_to_landing_mapper = city_to_landing_mapper

    def fetch(self, fetchers=None):
        to_slug = self.params.get('to_slug')
        national_version = self.params.get('national_version')
        to_settlement = self.cache_root.settlement_cache.get_settlement_by_slug(to_slug)
        self._validate(to_slug, to_settlement, national_version)

        to_id = to_settlement.Id

        fetcher = Fetcher(finish_callback=self.on_response)
        fetchers = []

        logger.info('Getting city landing for to_id: %s, nv: %s', to_id, national_version)

        fetcher.waiting_fields.add(self._FIELDS.statistics)
        fetchers.append(
            CityToLandingDataFetcher(
                field=self._FIELDS.statistics,
                to_id=to_id,
                national_version=national_version,
            )
        )

        fetcher.fetch(fetchers)

    def _validate(self, to_slug, to_settlement, national_version):
        not_found_error_message_parts = []
        if not national_version:
            not_found_error_message_parts.append('Empty national_version')
        if not to_settlement:
            not_found_error_message_parts.append('No settlement for to_slug {}'.format(to_slug))

        if not_found_error_message_parts:
            raise HTTPError(404, '\n'.join(not_found_error_message_parts))

    def on_response(self, data):
        # type: (Dict) -> None
        statistics = data.get(self._FIELDS.statistics)
        mapped_landing = self._city_to_landing_mapper.map(statistics=statistics)
        self.finish_callback(mapped_landing, field=self.field)
