# -*- encoding: utf-8 -*-
import logging
from enum import Enum
from datetime import date, timedelta
from typing import Dict, Optional

from travel.avia.api_gateway.application.cache.cache_root import CacheRoot
from travel.avia.api_gateway.lib.landings.templater import LandingTemplater
from travel.avia.api_gateway.application.utils import MONTHS
from travel.avia.api_gateway.settings import TRAVEL_FRONTEND_URL_RU
from travel.avia.library.python.enum.currency import Currency


logger = logging.getLogger(__name__)


class CityToLandingMapper:
    MAX_TOP_AIRLINES = 10
    CITY_LANDING_IMAGE_URL = 'https://avatars.mds.yandex.net/get-avia/233213/2a0000015a80437fec5bb0af3293c5f3b4b9/orig'
    CITY_LANDING_IMAGE_SIZE = (1815, 1212)
    BREADCRUMBS_TITLE_LINK = '{}/avia/'.format(TRAVEL_FRONTEND_URL_RU)

    def __init__(self, cache_root, templater):
        # type: (CacheRoot, LandingTemplater) -> None
        self._cache_root = cache_root
        self._templater = templater

    def map(self, statistics):
        # type: (dict) -> dict
        if not statistics:
            return {}
        to_settlement = self._cache_root.settlement_cache.get_settlement_by_id(statistics['toId'])
        to_point_key = _settlement_id_to_point_key(to_settlement.Id)
        faq_data = self._build_faq_data(to_settlement)
        faq_question_answers = self._build_faq_question_answers(faq_data)
        min_price = None
        return {
            'seoInfo': self._seo_info(to_settlement, min_price, faq_question_answers),
            'blocks': [
                _f
                for _f in [
                    self._breadcrumbs_block(to_settlement),
                    self._search_form(to_settlement, to_point_key),
                    self._dynamics_block(to_settlement, to_point_key),
                    self._best_to_fly_block(
                        to_settlement,
                        statistics.get('monthAndYearPrices'),
                    ),
                    self._popular_directions_block(to_settlement, directions=statistics.get('routeCrosslinks')),
                    self._closest_cities_block(
                        to_settlement,
                        statistics.get('nearestCities'),
                    ),
                    self._faq_block(faq_data, faq_question_answers),
                ]
                if _f
            ],
        }

    def _seo_info(self, to_settlement, min_price, faq_question_answers):
        title = self._templater.render(
            'seo.title',
            city_to=to_settlement,
            min_price=min_price,
        )
        description = self._templater.render(
            'seo.description',
            city_to=to_settlement,
        )
        return {
            'title': title,
            'description': description,
            'openGraph': {
                'title': title,
                'description': description,
                'image': self.CITY_LANDING_IMAGE_URL,  # Todo: тут хотели картинку города, если есть
                'imageSize': {
                    'width': self.CITY_LANDING_IMAGE_SIZE[0],
                    'height': self.CITY_LANDING_IMAGE_SIZE[1],
                },
            },
            'schemaOrg': {
                'faqItems': [
                    {
                        'question': q,
                        'answer': a,
                    }
                    for q, a in faq_question_answers
                ],
            },
        }

    def _breadcrumbs_block(self, to_settlement):
        return {
            'type': ERouteBlockType.BREADCRUMBS_BLOCK,
            'data': {
                'items': [
                    {
                        'title': self._templater.render('breadcrumbs.main_title'),
                        'link': self.BREADCRUMBS_TITLE_LINK,
                    },
                    {
                        'title': self._templater.render('breadcrumbs.city_title', city_to=to_settlement),
                    },
                ]
            },
        }

    def _search_form(self, to_settlement, to_point_key):
        return {
            'type': ERouteBlockType.SEARCH_FORM_BLOCK,
            'data': {
                'title': self._templater.render('search_form.title', city_to=to_settlement),
                'imageUrl': self.CITY_LANDING_IMAGE_URL,
                'searchFormParams': {
                    'isCalendarOpen': True,
                    'toId': to_point_key,
                },
            },
        }

    def _dynamics_block(self, to_settlement, to_point_key):
        return {
            'type': ERouteBlockType.DYNAMICS_BLOCK,
            'data': {
                'title': self._templater.render('dynamic.title', city_to=to_settlement),
                'navigationTitle': self._templater.render('dynamic.navigation'),
                'description': self._templater.render('dynamic.description', city_to=to_settlement),
                'toId': to_point_key,
                'when': (date.today() + timedelta(days=3)).strftime('%Y-%m-%d'),
            },
        }

    def _best_to_fly_block(self, to_settlement, prices_info):
        if not prices_info:
            return None
        if prices_info['yearMedianPrice'] == 0:
            return None
        title = self._templater.render(
            'best_to_fly.title',
            city_to=to_settlement,
        )
        return {
            'type': ERouteBlockType.SECTION_TEXT_BLOCK,
            'data': {
                'title': title,
                'navigationTitle': self._templater.render('best_to_fly.navigation'),
            },
            'children': [
                _f
                for _f in [
                    self._min_price_month_block(prices_info),
                    self._max_price_month_block(prices_info),
                ]
                if _f
            ],
        }

    def _min_price_month_block(self, prices_info):
        min_price_month_text = self._templater.render(
            'best_to_fly.min_price_month',
            min_price_month=MONTHS[prices_info['minMonth'] - 1],
            min_price_month_percentage=_calc_percent_lower(
                min_price=prices_info['minMonthMedianPrice'],
                avg_price=prices_info['yearMedianPrice'],
            ),
        )
        return {
            'type': ERouteBlockType.TEXT_BLOCK,
            'children': [
                {
                    'type': ERouteBlockType.PLAIN_TEXT_BLOCK,
                    'data': {
                        'text': min_price_month_text,
                    },
                },
            ],
        }

    def _max_price_month_block(self, prices_info):
        max_price_month_text = self._templater.render(
            'best_to_fly.max_price_month',
            max_price_month=MONTHS[prices_info['maxMonth'] - 1],
        )
        return {
            'type': ERouteBlockType.TEXT_BLOCK,
            'children': [
                {
                    'type': ERouteBlockType.PLAIN_TEXT_BLOCK,
                    'data': {
                        'text': max_price_month_text,
                    },
                },
            ],
        }

    def _popular_flights_block(self, to_settlement):
        title = self._templater.render(
            'popular_flights.title',
            city_to=to_settlement,
        )
        return {
            'type': ERouteBlockType.SECTION_TEXT_BLOCK,
            'data': {
                'title': title,
                'text': self._templater.render('popular_flights.text'),
                'navigationTitle': self._templater.render('popular_flights.navigation'),
            },
        }

    def _popular_directions_block(self, to_settlement, directions):
        if not directions:
            return None
        title = self._templater.render(
            'popular_directions.title',
            city_to=to_settlement,
        )
        return {
            'type': ERouteBlockType.CITY_DIRECTIONS_BLOCK,
            'data': {
                'title': title,
                'text': self._templater.render('popular_directions.text'),
                'navigationTitle': self._templater.render('popular_directions.navigation'),
                'items': [
                    {
                        'fromSlug': self._cache_root.settlement_cache.get_slug_by_id(direction['crosslinkFromId']),
                        'name': self._cache_root.settlement_cache.get_settlement_by_id(
                            direction['crosslinkFromId']
                        ).Title.Ru.Nominative,
                        'price': _map_price(direction),
                    }
                    for direction in directions
                ],
            },
        }

    def _closest_cities_block(self, to_settlement, closest_cities):
        if not closest_cities:
            return None
        return {
            'type': ERouteBlockType.CLOSEST_CITIES_BLOCK,
            'data': {
                'title': self._templater.render('closest.title', city_to=to_settlement),
                'navigation': self._templater.render('closest.navigation', city_to=to_settlement),
                'description': self._templater.render('closest.description', city_to=to_settlement),
                'items': [
                    {
                        'toSlug': self._cache_root.settlement_cache.get_slug_by_id(city['id']),
                        'name': self._cache_root.settlement_cache.get_settlement_by_id(city['id']).Title.Ru.Nominative,
                        'when': city['date'],
                        'price': _map_price(city),
                    }
                    for city in closest_cities
                ],
            },
        }

    def _build_faq_question_answers(self, faq_data):
        ready_faq_numbers = [3]
        question_answers = []
        for part_number in ready_faq_numbers:
            question = self._templater.render('faq.{}_q'.format(part_number), data=faq_data)
            answer = self._templater.render('faq.{}_a'.format(part_number), data=faq_data)
            if not question or not answer:
                continue
            question_answers.append((question, answer))
        return question_answers

    def _faq_block(self, faq_data, faq_question_answers):
        faq_parts = []
        for question, answer in faq_question_answers:
            faq_parts.append(
                {
                    'type': ERouteBlockType.SPOILER_BLOCK,
                    'data': {
                        'title': question,
                        'description': {
                            'type': ERouteBlockType.TEXT_BLOCK,
                            'children': [
                                {
                                    'type': ERouteBlockType.PLAIN_TEXT_BLOCK,
                                    'data': {
                                        'text': answer,
                                    },
                                },
                            ],
                        },
                    },
                }
            )

        return {
            'type': ERouteBlockType.SECTION_TEXT_BLOCK,
            'data': {
                'title': self._templater.render('faq.title', city_to=faq_data.city_to),
                'navigationTitle': self._templater.render('faq.navigation'),
            },
            'children': faq_parts,
        }

    def _build_airports(self, airport_ids):
        return [
            {
                'title': self._cache_root.station_cache.get_station_by_id(_id).TitleDefault,
                'iata': self._cache_root.station_code_cache.get_station_code_by_id(_id),
                'distance': 100,  # TODO: Посчитать расстояние
            }
            for _id in airport_ids
        ]

    def _build_airlines(self, airline_ids):
        return [
            {'title': self._cache_root.company_cache.get_company_by_id(airline_id).TitleRu}
            for airline_id in airline_ids
        ]

    def _build_faq_data(self, to_settlement):
        return FaqData(to_settlement)


class FaqData:
    def __init__(self, to_settlement):
        self.city_to = to_settlement


def _settlement_id_to_point_key(_id):
    return 'c{}'.format(_id)


def _calc_percent_lower(min_price, avg_price):
    return int(100.0 - 100.0 * min_price / avg_price)


def _map_price(object_info):
    # type: (Dict) -> Optional[Dict]
    if not (object_info.get('price') and object_info.get('currency')):
        return None
    return {
        'value': object_info['price'],
        'currency': Currency.from_str_with_correction(object_info['currency']),
    }


class ERouteBlockType(str, Enum):
    BREADCRUMBS_BLOCK = 'IBreadCrumbsBlock'
    SEARCH_FORM_BLOCK = 'ISearchFormBlock'
    TEXT_BLOCK = 'ITextBlock'
    SECTION_TEXT_BLOCK = 'ISectionTextBlock'
    PLAIN_TEXT_BLOCK = 'IPlainTextBlock'
    DYNAMICS_BLOCK = 'IDynamicsBlock'
    CLOSEST_CITIES_BLOCK = 'IClosestCitiesBlock'
    CITY_DIRECTIONS_BLOCK = 'ICityDirectionsBlock'
    SPOILER_BLOCK = 'ISpoilerBlock'
