from cachetools import TTLCache
from functools import cached_property
from typing import Optional

from google.protobuf.json_format import MessageToJson
from tornado.httpclient import HTTPResponse
from urllib.parse import urljoin
import tornado.escape

from travel.marketing.content.v1.api_pb2 import GetContentByIdRequest

from travel.avia.api_gateway.application.fetcher import Fetcher

_content_backend_city_image_ttl = 10 * 60  # 10 minutes
_content_backend_city_image_cache_size = 1024


class ContentBackendBaseFetcher(Fetcher):
    service: str = 'content-backend'
    CONTENT_BACKEND_API_URL = 'http://content-backend.production.travel.yandex.net/'

    @cached_property
    def content_backend_url(self):
        return self.params.get('content_backend_api_url') or self.CONTENT_BACKEND_API_URL


class ContentBackendCityImageFetcher(ContentBackendBaseFetcher):
    service: str = 'content-backend-geo-city-image'
    CONTENT_BACKEND_CITY_IMAGE_URL = '/grpc/geo/city/image'

    _cache_ = TTLCache(maxsize=_content_backend_city_image_cache_size, ttl=_content_backend_city_image_ttl)

    def __init__(self, make_request_sync: bool = False, *args, **kwargs):
        super(ContentBackendCityImageFetcher, self).__init__(*args, **kwargs)
        self._make_request_sync = make_request_sync

    @cached_property
    def city_image_path(self):
        return self.params.get('content_backend_city_image_url') or self.CONTENT_BACKEND_CITY_IMAGE_URL

    @cached_property
    def city_image_url(self):
        return urljoin(
            self.content_backend_url,
            self.city_image_path,
        )

    def fetch(self, fetchers: Optional[list['Fetcher']] = None) -> None:
        settlement_id = self.params['settlement_id']
        if (
            (data := ContentBackendCityImageFetcher._cache_.get(settlement_id)) or
            settlement_id in ContentBackendCityImageFetcher._cache_
        ):
            return self.finish_callback(data, field=self.field)
        req = GetContentByIdRequest(id=settlement_id)
        body = MessageToJson(req)
        if self._make_request_sync:
            response = self.request_sync(
                self.city_image_url,
                body=body,
                method='POST',
            )
            self.on_response(response)
        else:
            self.request(
                self.city_image_url,
                callback=self.on_response,
                body=body,
                raise_error=False,
                method='POST',
            )

    def on_response(self, response: HTTPResponse):
        if response.code == 404:
            ContentBackendCityImageFetcher._cache_[self.params['settlement_id']] = None
            return self.finish_callback(None, field=self.field)
        self.raise_for_status(response)
        data = tornado.escape.json_decode(response.body)
        if 'image_url' not in data:
            return self.finish_callback(None, field=self.field)
        ContentBackendCityImageFetcher._cache_[self.params['settlement_id']] = data
        return self.finish_callback(data, field=self.field)
